package pgdlutil;

our $VERSION = '0.0.0';
use warnings;
use strict;
no strict 'refs';
use Carp;
$Carp::Verbose = 0;
use POSIX 'floor';

require Exporter;
our @ISA = qw(Exporter);

our @EXPORT = qw(
                    pg2_parse_pgdl
                    pg2_dump_parsed_pgdl
                    pg2_get_parameter
                    pg2_get_modules
                    pg2_read_template
                    pg2_write_file
                    pg2_foreach_ioport_of_type
                    pg2_verbose
                    pg2_debug
                    width_of
               );

my %Constant        = ();
my %Parameter       = ();
my %Module          = ();
my %Signal          = ();

my $Rootpath        = ''; # pgpg2 root path specified by
                          # environment variable 'PGPG2PATH' (eg. /usr/local/pgpg2/)
my $Verbose     = 0;
my $Debug       = 0;

my $Arch             = undef;
my $Backendid        = undef;
my $Chipid           = undef;
my $Delay            = undef;
my $Headerpath       = undef;
my $Libpath          = undef;
my $Jmemsize         = undef;
my $Npipe            = undef;
my $Prefix           = undef;
my $Use_fixed_chipid = undef;
my $Ipset            = undef;
my $Jpset            = undef;
my $Coeffset         = undef;
my $Foset            = undef;

my %Arranged_parameter = (
    ARCH             => \$Arch,
    BACKENDID        => \$Backendid,
    CHIPID           => \$Chipid,
    DELAY            => \$Delay,
    HEADERPATH       => \$Headerpath,
    LIBPATH          => \$Libpath,
    JMEMSIZE         => \$Jmemsize,
    NPIPE            => \$Npipe,
    PREFIX           => \$Prefix,
    USE_FIXED_CHIPID => \$Use_fixed_chipid,
    IPSET            => \$Ipset,
    JPSET            => \$Jpset,
    COEFFSET         => \$Coeffset,
    FOSET            => \$Foset,
);

##########################################
#
# functions to be exported
#

# Parse given .pgdl file.  And then register parameters, modules,
# instances, and signals.
#
sub pg2_parse_pgdl {
    my ($infile) = @_;

    _set_root_path();

    my $intext = _read_pgdl($infile);
    my $intext_save = $intext;

    _dprintf("\ntext read, preprocessed, and comment stripped: --------\n");
    _dprintf($intext);
    _dprintf("--------------------------------------------------------\n\n");

    # register constatns
    while ($intext) {
        if ($intext =~ /\#define\s+([a-zA-Z]\w*)\s+(\S*)/) {
            _register_constant($1, $2);
            $intext = $` . $';
        }
        else {
            $intext = '';
        };
    }

    # register parameters
    $intext = $intext_save;
    while ($intext) {
        if ($intext =~ / \/  ([a-zA-Z]\w*) \s*(.*)\s* /x) {
            # remove trailing space(s) and register.
            _register_parameter($1,
                               map {$_ =~ s/\s+$//g; $_;} split(/\s*,\s*/, $2));
            $intext = $` . $';
        }
        else {
            $intext = '';
        };
    }

    $intext = $intext_save;
    while ($intext) {
        if ($intext =~ /(  pg_ [^\(]*  )  \(  ([^\)]*)  \)/xs) {
            _register_module($1,
                             map {$_ =~ s/\s+$//g; $_;} split(/\s*,\s*/, $2));
            $intext = $` . $';
        }
        else {
            $intext = '';
        };
    }

    # substitue constants in %Parameter
    #
    _dprintf("\n%%Constant\n");
    for (keys %Constant) {
        _dprintf("$_ => $Constant{$_}\n");
    }

    _dprintf("\n%%Parameter before substitution by %%Constant\n");
    for my $paramset (keys %Parameter) {
        local $" = '|';
        for my $param (@{$Parameter{$paramset}}) {
            _dprintf("$paramset => @{$param}\n");
        }
    }

    for my $paramset (keys %Parameter) {
        for my $param (@{$Parameter{$paramset}}) {
            for my $arg (@{$param}) {
                if ($Constant{$arg}) {
                    $arg = $Constant{$arg};
                }
            }
        }
    }

    _dprintf("\n%%Parameter after substitution by %%Constant\n");
    for my $paramset (keys %Parameter) {
        local $" = '|';
        for my $param (@{$Parameter{$paramset}}) {
            _dprintf("$paramset => @{$param}\n");
        }
    }

    # dump registered modules and signals
    #
    _dprintf("\n%%Module\n");
    for my $module (values %Module) {
        _dprintf("name: " . $module->{name} . "\n");
    }

    _arrange_parameters();

    return $intext_save;
}

#
# dump all parameters and modules parsed
# (just for debugging purpose).
#
sub pg2_dump_parsed_pgdl {

    # scalar parameters
    #
    my @params = (
                  'ARCH',       'BACKENDID', 'CHIPID',   'DELAY',
                  'JMEMSIZE',  'NPIPE',    'USE_FIXED_CHIPID',
                  );
    _dprintf("=== parameters:\n");
    for my $name (@params) {
        my $val = pg2_get_parameter($name);
        if (! defined $val) {
            $val = 'undef';
            _vprintf("Parameter '$name' is not defined.\n");
        }
        _dprintf("  %-16s : " . $val . "\n", $name);
    }

    # vector parameters
    #
    @params = (
               'HEADERPATH', 'LIBPATH',
               );
    for my $paramname (@params) {
        my $values = pg2_get_parameter($paramname);

        next if ! defined $values;

        _dprintf("  %-17s: ", "${paramname}");
        for my $val (@{$values}) {
            _dprintf("$val |");
        }
        _dprintf("\n");
    }

    # structured parameters
    #
    for my $paramname ('IPSET', 'JPSET', 'COEFFSET', 'FOSET') {
        my $params = pg2_get_parameter($paramname);

        next if ! defined $params;

        my $i = 0;
        for my $param (@{$params}) {
            _dprintf("  %-17s: ", "${paramname}[$i]");
            for my $val (@{$param}) {
                _dprintf("$val |");
            }
            _dprintf("\n");
            $i++;
        }
    }

    _dprintf("\n\n");
    _dprintf("=== modules:\n");
    for my $m (values %{pg2_get_modules()}) {
        _dprintf("  module: $m->{name}\n");
        my $i = 0;
        for my $args (@{$m->{argslist}}) {
            _dprintf("    args of instance$i: @{$args}\n");
            $i++;
        }
    }
    _dprintf("\n\n");
}



#
# applies &{$do_func}() for each I/O port of type 'JPSET',
# and then invokes &{$return_func}() to returns its returning value.
#
sub _foreach_jpport {
    my ($do_func, $return_func) = @_;
    my $outtext   = '';
    my $pjoff     = 0;    # bit-offset of 'p_jdata'.
    my $nsignal   = 0;
    my $io = 'JPSET';

    for my $param (@{$Parameter{$io}}) { # loop for each signal

        # parse I/O parameters.
        #
        my ($var, $ctype, $ne, $type) = @{$param};
        my ($w, $m, $scale, $offset);
        $ne =~ s/.*\[(\d+)\]/$1/; # number of elements
        for ($type) {
            /int/ and do {
                ($w, $scale, $offset) = @{$param}[4..6];
                last;
            };
            /float|log/ and do {
                ($w, $m, $scale) = @{$param}[4..6];
                last;
            };
        }

        for my $i ( 0 .. $ne - 1 ) {    # loop for each signal element
            my $isfirst = 0;
            if ($i == 0 && $nsignal == 0) {
                $isfirst = 1;
            }
            my $islast = 0;
            if ($i == $ne - 1 && $nsignal == scalar @{$Parameter{$io}} - 1) {
                $islast = 1;
            }
            $outtext .= &{$do_func}(
                {
                    var       => $var,
                    ctype     => $ctype,
                    ne        => $ne,
                    type      => $type,
                    width     => $w,
                    mantissa  => $m,
                    scale     => $scale,
                    offset    => $offset,
                    index     => $i,
                    signaloff => 0,
                    pjoff     => $pjoff,
                    nsignal   => $nsignal,
                    isfirst   => $isfirst,
                    islast    => $islast,
                }
            );
            $pjoff += $w;

        }    # loop for each signal element
        $nsignal++;
    }    # loop for each signal

    return &{$return_func}(
        {
            outtext   => $outtext,
            pjoff     => $pjoff,
            nsignal   => $nsignal,
        }
    );
}

#
# applies &{$do_func}() for each I/O port of type 'JPSET', 'IPSET', 'COEFFSET', or 'FOSET',
# and then invokes &{$return_func}() to returns its returning value.
#
sub pg2_foreach_ioport_of_type {
    my ($io, $do_func, $return_func) = @_;
    my $outtext   = '';
    my $pdoff     = 0;    # bit-offset of 'p_datai' or 'p_datao'.
    my $adr       = 0;
    my $nsignal   = 0;
    my $buswidth  = 0;

  IO_TYPE:
    for ($io) {
        /JPSET/ and do {
            return _foreach_jpport($do_func, $return_func);
            last IO_TYPE;
        };
        /IPSET | FOSET/x and do {
            $buswidth = 64;
            last IO_TYPE;
        };
        /COEFFSET/ and do {
            $buswidth = 24;
            last IO_TYPE;
        };
        do {
            croak "unknown io type:$io.\n";
            last;
        };
    }

    for my $param (@{$Parameter{$io}}) { # loop for each signal

        # parse I/O parameters.
        #
        my ($var, $ctype, $ne, $type) = @{$param};
        my ($w, $m, $scale, $offset);
        $ne =~ s/.*\[(\d+)\]/$1/; # number of elements
        for ($type) {
            /int/ and do {
                ($w, $scale, $offset) = @{$param}[4..6];
                last;
            };
            /float|log/ and do {
                ($w, $m, $scale) = @{$param}[4..6];
                last;
            };
        }

        for my $i (0 .. $ne-1) { # loop for each signal element
            my $sigoff = 0; # bits already processed.

=pod
            # align to bit boundary
            my @boundaries = (64, 32);
            if ($io eq 'COEFFSET') {
                @boundaries = (24);
            }
            for my $align (@boundaries) {
                if ($w >= $align) {
                    $pdoff = (floor(($pdoff - 1) / $align) + 1) * $align;
                }
            }
            # avoid bit-assignment across $buswidth-bit boundary
            if ($w <= $buswidth) {
                my $boundary = (floor(($pdoff - 1) / $buswidth) + 1) * $buswidth;
                if ($boundary < $pdoff + $w) {
                    $pdoff = $boundary;
                }
            }
=cut

            # loop for a bit slice
            while ($sigoff < $w) {
                if ($pdoff == $buswidth) {
                    $pdoff = 0;
                    $adr++;
                }
                my $slicewidth = $buswidth; # number of bits to be processed inside this loop
                if ($sigoff + $slicewidth > $w) { # end of a variable $var.
                    $slicewidth = $w - $sigoff;
                }
                if ($pdoff + $slicewidth > $buswidth) { # end of data at an address $adr.
                    $slicewidth = $buswidth - $pdoff;
                }
		my $isfirst = 0;
		if ($sigoff == 0 && $i == 0 && $nsignal == 0) {
		    $isfirst = 1;
		}
                my $islast = 0;
                if ($sigoff + $slicewidth >= $w &&
                    $i == $ne - 1 &&
                    $nsignal == scalar @{$Parameter{$io}} - 1) {
                    $islast = 1;
                }
    
                $outtext .= &{$do_func}(
                    {
                        var        => $var,
                        ctype      => $ctype,
                        ne         => $ne,
                        type       => $type,
                        width      => $w,
                        mantissa   => $m,
                        scale      => $scale,
                        offset     => $offset,
                        buswidth   => $buswidth,
                        index      => $i,
                        adr        => $adr,
                        signaloff  => $sigoff,
                        slicewidth => $slicewidth,
                        pdoff      => $pdoff,
                        nsignal    => $nsignal,
                        isfirst    => $isfirst,
                        islast     => $islast,
                    }
                );

                $sigoff += $slicewidth;
                $pdoff  += $slicewidth;

            } # loop for a bit slice
        } # loop for each signal element
        $nsignal++;
    } # loop for each signal

    return &{$return_func}(
        {
            outtext   => $outtext,
            buswidth  => $buswidth,
            adr       => $adr,
            nsignal   => $nsignal,
        }
    );
}


# read a template file in $Rootpath/src/templates/
#
# usage:
#   pg2_read_template(file_name_base[, file_name_extention])
#   to read  $Rootpath/src/templates/file_name_base.template.file_name_extention
#
sub pg2_read_template {
    local (*IN);
    local ($/);
    my ($fname, $ext) = @_;
    if (!defined $ext) {
        $ext = 'vhd';
    }
    open(IN, "< $Rootpath/src/templates/$fname.template." . $ext) or croak "Cannot open $fname\n";
    my $intext = <IN>;
    close IN;
    return $intext;
}

# return value(s) of a parameter given by the name.
#
sub pg2_get_parameter {
    my ($name) = @_;
    my $val = ${$Arranged_parameter{$name}};
    return $val;
}

# return all modules registered.
#
sub pg2_get_modules {
    return \%Module;
}

# write $outtext to a file $fname.
#
sub pg2_write_file {
    my ($fname, $outtext) = @_;
    croak "Cannot open $fname\n" if !open(OUT, "> $fname");
    print OUT "$outtext";
    close OUT;
    return;
}

# returns necessary bit width to express $value.
# used by perl text embedded in the template files.
#
sub width_of {
    my ($value) = @_;
    my $width;

    if ((abs $value) > 1) {
        $width = int log((abs $value) - 1) / log(2) + 1;
    }
    else {
        $width = 1;
    }
    return $width;
}

sub pg2_verbose {
    my ($val) = @_;

    if (! defined $val) {
        $val = 1;
    }

    $Carp::Verbose = $val;
    $Verbose       = $val;
}

sub pg2_debug {
    my ($val) = @_;

    if (! defined $val) {
        $val = 1;
    }

    $Carp::Verbose = $val;
    $Debug         = $val;
}

#############################################################
#
# module-internal utilities
#


# arrange raw parameters extracted from .pgdl
# so that they can be easily read & used.
#
# - remove unnecessary white spaces, double quotes, commas.
# - remove array hierarchy of a parameter which should appear
#   only once in a .pgdl file, such as NPIPE and JMEMSIZE.
#
sub _arrange_parameters {

    # Arch
    my $arch_name = $Parameter{ARCH}->[0]->[0];
    $arch_name =~ s/\"//g;
    $Arch = _get_arch($arch_name);

    # Backendid
    my $bid = $Parameter{BACKENDID}->[0]->[0];
    $Backendid = defined $bid ? $bid : '0';

    # Chipid & Use_fixed_chipid
    ($Use_fixed_chipid, $Chipid) = _get_chipid();

    # Delay
    $Delay = $Parameter{DELAY}->[0]->[0];
    $Delay =~ s/[^\d]*(\d+)[^\d]*/$1/g;

    # JMEMSIZE
    $Parameter{JMEMSIZE}->[0]->[0] =~ s/[^\d]*(\d+)[^\d]*/$1/g;
    my $jms = $Parameter{JMEMSIZE}->[0]->[0];
    $Jmemsize = 1 << (int (log($jms - 1) / log(2)) + 1); # round up to powers of two.

    if ($jms != $Jmemsize) {
        _vprintf("JMEMSIZE $jms is rounded up to $Jmemsize.\n");
    }

    # NPIPE
    $Parameter{NPIPE}->[0]->[0] =~ s/[^\d]*(\d+)[^\d]*/$1/g;
    $Npipe = $Parameter{NPIPE}->[0]->[0];

    # PREFIX
    my $prefix = $Parameter{PREFIX}->[0]->[0];
    if (defined $prefix) {
        $prefix =~ s/\"//g;
    }
    else {
        $prefix = 'pg2';
    }
    $Prefix = $prefix;

    # HEADERPATH
    my @headerpath = ( q{.}, $Rootpath . '/include');
    if (defined $Parameter{HEADERPATH}) {
        for (@{$Parameter{HEADERPATH}->[0]}) {
            my $apath = $_;
            $apath =~ s/\"//g;
            push @headerpath, $apath;
        }
    }
   $Headerpath = \@headerpath;

    # LIBPATH
#    my @libpath = ( q{.}, $Rootpath . '/lib');
    my @libpath = ( q{.} );
    if (defined $Parameter{LIBPATH}) {
        for (@{$Parameter{LIBPATH}->[0]}) {
            my $apath = $_;
            $apath =~ s/\"//g;
            push @libpath, $apath;
        }
    }
    $Libpath = \@libpath;

    # IPSET
    $Ipset = $Parameter{IPSET};

    # JPSET
    $Jpset = $Parameter{JPSET};

    # COEFFSET
    $Coeffset = $Parameter{COEFFSET};

    # FOSET
    $Foset = $Parameter{FOSET};
}

# returns value of parameters 'USE_FIXED_CHIPID' & 'CHIPID'
#
sub _get_chipid {
    my $use_fixed;
    my $chipid;

    for my $param (@{$Parameter{'USE_FIXED_CHIPID'}}) {
        $use_fixed = $param->[0];
    }
    if (!defined $use_fixed) {
        $use_fixed = 0; # set 0 by default.
    }
    for my $param (@{$Parameter{'CHIPID'}}) {
        $chipid = $param->[0];
    }
    if ($use_fixed) {
        if (!defined $chipid) {
            croak "Parameter 'USE_FIXED_CHIPID' is set to 1, " .
              "but Parameter 'CHIPID' not set.\n";
        }
        if ($chipid < 1) {
            croak "Too small 'CHIPID': $chipid.\n";
        }
        elsif ($chipid > $Arch->{nchip}) {
            croak "Too large 'CHIPID': $chipid.\n";
        }
    }
    return ($use_fixed, $chipid);
}

# returns architecture dependent parameters
#
sub _get_arch {
    my ($arch_name) = @_;
    my $g7m100 = {
        name        => 'g7m1',       # name passed to gwrap.
        productid   => 1,
        modelid     => 1,
        device      => 'Cyclone II', # device family name used by QuartusII.
        nchip       => 1,            # number of pFPGA chips.
        reduce_fout => 0,            # set 1 if 'fout's from multiple pFPGAs are summed up automatically.
        ipin_depth  => 8,            # ipin address width
        cin_depth   => 8,            # cin  address width
        fout_depth  => 8,            # fout address width
    };
    my $g7m300p1 = {
        name        => 'g7p1',       # name passed to gwrap.
        productid   => 1,
        modelid     => 3,
        device      => 'Cyclone II',
        nchip       => 3,
        reduce_fout => 1,
        ipin_depth  => 8,            # ipin address width
        cin_depth   => 8,            # cin  address width
        fout_depth  => 8,            # fout address width
    };
    my $g7m300p2 = {
        name        => 'g7p2',       # name passed to gwrap.
        productid   => 1,
        modelid     => 3,
        device      => 'Cyclone II',
        nchip       => 3,
        reduce_fout => 1,
        ipin_depth  => 8,            # ipin address width
        cin_depth   => 8,            # cin  address width
        fout_depth  => 8,            # fout address width
    };
    my $g7m300p6 = {
        name        => 'g7p6',       # name passed to gwrap.
        productid   => 1,
        modelid     => 3,
        device      => 'Cyclone II',
        nchip       => 3,
        reduce_fout => 1,
        ipin_depth  => 8,            # ipin address width
        cin_depth   => 8,            # cin  address width
        fout_depth  => 8,            # fout address width
    };
    my $g7m600p1 = {
        name        => 'g7p1',       # name passed to gwrap.
        productid   => 1,
        modelid     => 6,
        device      => 'Cyclone II',
        nchip       => 6,
        reduce_fout => 1,
        ipin_depth  => 8,            # ipin address width
        cin_depth   => 8,            # cin  address width
        fout_depth  => 8,            # fout address width
    };
    my $g7m600p2 = {
        name        => 'g7p2',       # name passed to gwrap.
        productid   => 1,
        modelid     => 6,
        device      => 'Cyclone II',
        nchip       => 6,
        reduce_fout => 1,
        ipin_depth  => 8,            # ipin address width
        cin_depth   => 8,            # cin  address width
        fout_depth  => 8,            # fout address width
    };
    my $g7m600p6 = {
        name        => 'g7p6',       # name passed to gwrap.
        productid   => 1,
        modelid     => 6,
        device      => 'Cyclone II',
        nchip       => 6,
        reduce_fout => 1,
        ipin_depth  => 8,            # ipin address width
        cin_depth   => 8,            # cin  address width
        fout_depth  => 8,            # fout address width
    };
    my $g7m800 = {
        name        => 'g7m8',       # name passed to gwrap.
        productid   => 1,
        modelid     => 8,
        device      => 'Cyclone III', # device family name used by QuartusII.
        nchip       => 1,            # number of pFPGA chips. m800 has 4 chips but they are handled independently.
        reduce_fout => 0,            # set 1 if 'fout's from multiple pFPGAs are summed up automatically.
        ipin_depth  => 8,            # ipin address width
        cin_depth   => 8,            # cin  address width
        fout_depth  => 8,            # fout address width
    };
    my $gdrtb2 = {
        name        => 'dr2',        # name passed to gwrap.
        productid   => 3,
        modelid     => 2,
        device      => 'Stratix GX',
        nchip       => 1,
        reduce_fout => 0,
    };
    my $gdrtb3 = {
        name        => 'dr3',        # name passed to gwrap.
        productid   => 3,
        modelid     => 3,
        device      => 'Arria GX',
        nchip       => 1,
        reduce_fout => 0,
    };
    my $gdrtb4 = {
        name        => 'dr4',        # name passed to gwrap.
        productid   => 3,
        modelid     => 4,
        device      => 'Arria GX',
        nchip       => 4,
        reduce_fout => 0,
    };
    my %arch_list = (
        GRAPE7M100   => $g7m100,
        GRAPE7M300P1 => $g7m300p1,
        GRAPE7M300P2 => $g7m300p2,
        GRAPE7M300P6 => $g7m300p6,
        GRAPE7M600P1 => $g7m600p1,
        GRAPE7M600P2 => $g7m600p2,
        GRAPE7M600P6 => $g7m600p6,
        GRAPE7M800   => $g7m800,
        GRAPEDRTB2   => $gdrtb2,
        GRAPEDRTB3   => $gdrtb3,
        GRAPEDRTB4   => $gdrtb4,
    );

    my $arch_info = $arch_list{$arch_name};

    if (!defined($arch_info)) {
        printf STDERR
          "invalid architecture name: $arch_name.\n" .
          "valid architecuture names are:\n";
        for (sort keys %arch_list) {
            printf STDERR "    $_\n";
        }
        die "\n";
    }

    return $arch_info;
}

sub _set_root_path {
    my $rootpath = $ENV{PGPG2PATH};
    if (defined $rootpath) {
        $Rootpath = $rootpath;
    }
    else {
        $Rootpath = q{.};
    }
}

sub _read_pgdl {
    local (*IN);
    local ($/); # slurp the input
    my ($fname) = @_;
    my $intext;

    if (!defined $fname) {
        $fname = q{-};
    }

#    croak "$!\n" if !open(IN, "cpp $fname | grep -v '\#' |");
#    $intext = <IN>;
#    close IN;

    croak "$!\n" if !open(IN, "<$fname");
    $intext = <IN>;
    close IN;
    # remove C-style comments (// and /* */)
    #
    $intext =~ s/(\/\/.*)//g;
    $intext =~ s/( \/\* .* \*\/ )//gxs; # s makes '.' matches '\n'.

    return $intext;
}

sub _register_module {
    my $name = shift @_;
    my @args = @_;
    my $module = $Module{$name};

    if (defined $module) {
        $module->{ninstance}++;
        push @{$module->{argslist}}, \@args;
    }
    else {
        $Module{$name} = {
            name      => $name,
            ninstance => 1,
            argslist  => [ \@args ],
        };
    }
}

sub _register_parameter {
    my $name = shift @_;
    my @args = @_;

    local $" = '|';
    _dprintf("parameter: $name\n");
    _dprintf("args: @args|\n\n");

    push @{$Parameter{$name}}, \@args;
}

sub _register_constant {
    my $name = shift @_;
    my $val = shift @_;

    _dprintf("constant: $name\n");
    _dprintf("value: |$val|\n\n");

    $Constant{$name} = $val;
}

sub _dprintf {
    if ($Debug) {
        printf STDERR @_;
    }
}

sub _vprintf {
    if ($Verbose) {
        printf STDERR @_;
    }
}



#
# functions not explicitly exported but might be invoked by functions
# out of this package.
#

sub _found_eta {
    my $found = 0;
    my $coeffs = pg2_get_parameter('COEFFSET');
    if (defined $coeffs && @{$coeffs} == 1) {
        # look for 'eta'
        my ($etaparam) = grep {$_->[0] eq 'eta'} @{$coeffs};
        if (!defined $etaparam) {
            return $found;
        }
        my ($var, $ctype, $ne) = @{$etaparam};
        $ne =~ s/.*\[(\d+)\]/$1/; # number of elements
        if ($ne != 1 || $ctype ne 'double') {
            return $found;
        }
        $found = 1;
    }
    return $found;
}

sub _found_xmj {
    my $found = 0;
    my $jps = pg2_get_parameter('JPSET');
    if (defined $jps && @{$jps} == 2) {
        # look for 'xj'
        my ($xjparam) = grep {$_->[0] eq 'xj'} @{$jps};
        if (!defined $xjparam) {
            return $found;
        }
        my ($xjvar, $xjctype, $xjne) = @{$xjparam};
        $xjne =~ s/.*\[(\d+)\]/$1/;
        if ($xjne != 3 || $xjctype ne 'double') {
            return $found;
        }

        # look for 'mj'
        my ($mjparam) = grep {$_->[0] eq 'mj'} @{$jps};
        if (!defined $mjparam) {
            return $found;
        }
        my ($mjvar, $mjctype, $mjne) = @{$mjparam};
        $mjne =~ s/.*\[(\d+)\]/$1/;
        if ($mjne != 1 || $mjctype ne 'double') {
            return $found;
        }

        $found = 1;
    }
    return $found;
}

sub _found_xi {
    my $found = 0;

    if (! _found_eps2()) {
        return $found;
    }

    my $ips = pg2_get_parameter('IPSET');
    if (defined $ips && @{$ips} == 2) {
        # look for 'xi'
        my ($xiparam) = grep {$_->[0] eq 'xi'} @{$ips};
        if (!defined $xiparam) {
            return $found;
        }
        my ($xivar, $xictype, $xine) = @{$xiparam};
        $xine =~ s/.*\[(\d+)\]/$1/;
        if ($xine != 3 || $xictype ne 'double') {
            return $found;
        }
        $found = 1;
    }
    return $found;
}

sub _found_eps2 {
    my $found = 0;
    my $ips = pg2_get_parameter('IPSET');
    if (defined $ips && @{$ips} == 2) {
        # look for 'eps2'
        my ($eps2param) = grep {$_->[0] eq 'eps2'} @{$ips};
        if (!defined $eps2param) {
            return $found;
        }
        my ($eps2var, $eps2ctype, $eps2ne) = @{$eps2param};
        $eps2ne =~ s/.*\[(\d+)\]/$1/;
        if ($eps2ne != 1 || $eps2ctype ne 'double') {
            return $found;
        }
        $found = 1;
    }
    return $found;
}

sub _found_a {
    my $found = 0;
    my $fos = pg2_get_parameter('FOSET');
    if (defined $fos && (@{$fos} == 1 || @{$fos} == 2)) {
        # look for 'a'
        my ($aparam) = grep {$_->[0] eq 'a'} @{$fos};
        if (!defined $aparam) {
            return $found;
        }
        my ($avar, $actype, $ane) = @{$aparam};
        $ane =~ s/.*\[(\d+)\]/$1/;
        if ($ane != 3 || $actype ne 'double') {
            return $found;
        }
        $found = 1;
    }
    return $found;
}

sub _found_p {
    my $found = 0;
    my $fos = pg2_get_parameter('FOSET');
    if (defined $fos && @{$fos} == 2) {
        # look for 'p'
        my ($pparam) = grep {$_->[0] eq 'p'} @{$fos};
        if (!defined $pparam) {
            return $found;
        }
        my ($pvar, $pctype, $pne) = @{$pparam};
        $pne =~ s/.*\[(\d+)\]/$1/;
        if ($pne != 1 || $pctype ne 'double') {
            return $found;
        }
        $found = 1;
    }
    return $found;
}

sub _generate_old_api_prototype {
    my $outtext = q{};
    my $indent = q{ } x 4;

    # generate 'set_eta' prototype if found 'eta'.
    if (_found_eta()) {
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_eta(double eta);\n";
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_etaMC(int devid, double eta);\n";
    }

    # generate 'set_xmj' prototype if found 'xj' & 'mj'.
    if (_found_xmj()) {
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_xmj(int adr, int nj, double (*xj)[3], double *mj);\n";
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_xmjMC(int devid, int adr, int nj, double (*xj)[3], double *mj);\n";
    }

    # generate 'set_xi' prototype if found 'xi'.
    if (_found_xi()) {
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_xi(int ni, double (*xi)[3]);\n";
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_xiMC(int devid, int ni, double (*xi)[3]);\n";
    }

    # generate 'get_force' prototype if found 'a'.
    # note that the prototype is generated even if 'p' may not exist.
    if (_found_a()) {
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_get_force(int ni, double (*a)[3], double *p);\n";
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_get_forceMC(int devid, int ni, double (*a)[3], double *p);\n";
    }

    # generate 'set_eps/eps2/eps_to_all/eps2_to_all' prototype if found 'eps2'.
    if (_found_eps2()) {
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_eps(int ni, double *eps);\n";
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_epsMC(int devid, int ni, double *eps);\n";

        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_eps2(int ni, double *eps2);\n";
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_eps2MC(int devid, int ni, double *eps2);\n";

        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_eps_to_all(double eps);\n";
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_eps_to_allMC(int devid, double eps);\n";

        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_eps2_to_all(double eps2);\n";
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_set_eps2_to_allMC(int devid, double eps2);\n";

    }

    # generate 'calculate_force_on_x' prototype if found 'xi', 'a'.
    if (_found_xi() && _found_a()) {
        $outtext .= $indent;
        $outtext .= "void   ${Prefix}_calculate_force_on_x(double (*xi)[3], double (*a)[3], double *p, int ni);\n";
    }

    if ($outtext) {
        $outtext = "    // for backward compatibility\n" . $outtext;
    }
    return $outtext;
}

sub _generate_old_api_definition {
    my ($intext) = @_; # $intext contains entire pgutil.template.c file.
    my $indent = q{ } x 4;
    my $outtext = $intext;
    my $match = 0;

    if (_found_p()) {
        $outtext =~ s/<OLD_API_P>/, p/g;
        $match = 1;
    }
    else {
        $outtext =~ s/<OLD_API_P>//g;
    }

    if (_found_eta()) {
        $outtext =~ s{<OLD_API_FOUND_ETA>\s*?\n(.*?)</OLD_API_FOUND_ETA>\s*?\n}{$1}xmsg;
        $match = 1;
    }
    else {
        $outtext =~ s{<OLD_API_FOUND_ETA>\s*?\n(.*?)</OLD_API_FOUND_ETA>\s*?\n}{}xmsg;
    }

    if (_found_xmj()) {
        $outtext =~ s{<OLD_API_FOUND_XMJ>\s*?\n(.*?)</OLD_API_FOUND_XMJ>\s*?\n}{$1}xmsg;
        $match = 1;
    }
    else {
        $outtext =~ s{<OLD_API_FOUND_XMJ>\s*?\n(.*?)</OLD_API_FOUND_XMJ>\s*?\n}{}xmsg;
    }

    if (_found_eps2()) {
        $outtext =~ s{<OLD_API_FOUND_EPS2>\s*?\n(.*?)</OLD_API_FOUND_EPS2>\s*?\n}{$1}xmsg;
        $match = 1;
    }
    else {
        $outtext =~ s{<OLD_API_FOUND_EPS2>\s*?\n(.*?)</OLD_API_FOUND_EPS2>\s*?\n}{}xmsg;
    }

    if (_found_a()) {
        $outtext =~ s{<OLD_API_FOUND_A>\s*?\n(.*?)</OLD_API_FOUND_A>\s*?\n}{$1}xmsg;
        $match = 1;
    }
    else {
        $outtext =~ s{<OLD_API_FOUND_A>\s*?\n(.*?)</OLD_API_FOUND_A>\s*?\n}{}xmsg;
    }
    if (_found_eps2() && _found_a()) {
        $outtext =~ s{<OLD_API_FOUND_EPS2_A>\s*?\n(.*?)</OLD_API_FOUND_EPS2_A>\s*?\n}{$1}xmsg;
        $match = 1;
    }
    else {
        $outtext =~ s{<OLD_API_FOUND_EPS2_A>\s*?\n(.*?)</OLD_API_FOUND_EPS2_A>\s*?\n}{}xmsg;
    }

    if ($match) {
        $outtext =~ s{<OLD_API_EXISTS>\s*\n(.*?)</OLD_API_EXISTS>\s*\n}{$1}xmsg;
    }
    else {
        $outtext =~ s{<OLD_API_EXISTS>\s*\n(.*?)</OLD_API_EXISTS>\s*\n}{}xmsg;
    }

    return $outtext;
}

# returns arguments of a function call to
# <prefix>_set_jpMC/_set_ipMC/_get_foutMC().
#
sub _generate_iocall {
    my ($io, $opt1) = @_;
    my $raw = 0;
    my $off = 0;
    if (defined $opt1 && $opt1 eq 'raw') {
        $raw = 1;
    }
    if (defined $opt1 && $opt1 eq 'off') {
        $off = 1;
    }

    my $do = sub {
        my ($args)   = @_;
        my $var      = $args->{var};
        my $sigoff   = $args->{signaloff};
        my $ne       = $args->{ne};
        my $index    = $args->{index};
        my $ctype    = $args->{ctype};
        my $outtext = q{};

        return q{} if $index > 0;
        return q{} if $sigoff > 0;

        for ($io) {
            /JPSET/ and do {
                if ($ne > 1) {
                    $outtext .= ", ($ctype (*)[$ne])(${var}[j0])";
                } else {
                    $outtext .= ", $var + j0";
                }
                last;
            };
            /IPSET|COEFFSET/ and do {
                if ($off) {
                    if ($ne == 1) {
                        $outtext .= ", &${var}[off]";
                    }
                    else {
                        $outtext .= ", (double (*)[3])${var}[off]";
                    }
                }
                else {
                    $outtext .= ", $var";
                }
                last;
            };

            /FOSET/ and do {
                if ($raw) {
                    $outtext .= ", i${var}";
                }
                elsif ($off) {
                    if ($ne == 1) {
                        $outtext .= ", &${var}[off]";
                    }
                    else {
                        $outtext .= ", (double (*)[3])${var}[off]";
                    }
                }
                else {
                    $outtext .= ", ${var}";
                }
                last;
            };
        }
        return $outtext;
    };

    my $return = sub {
        my ($args)   = @_;
        my $outtext  = $args->{outtext};
        return $outtext;
    };

    my $outtext = pg2_foreach_ioport_of_type($io, $do, $return);

    return $outtext;
}

# returns arguments of a library function
# <prefix>_set_jp/_set_ip/_get_fout().
#
sub _generate_ioargs {
    my ($io) = @_;

    my $do = sub {
        my ($args)   = @_;
        my $var      = $args->{var};
        my $sigoff   = $args->{signaloff};
        my $ne       = $args->{ne};
        my $index    = $args->{index};
        my $ctype    = $args->{ctype};
        my $isfirst  = $args->{isfirst};
        my $outtext  = q{};

        return q{} if $index > 0;
        return q{} if $sigoff > 0;

	if (!$isfirst) {
	  $outtext .= q{, };
	}
        if ($io =~ /COEFFSET/) {
            if ($ne > 1) {
                $outtext .= "${ctype} ${var}[$ne]";
            } else {
                $outtext .= "$ctype $var";
            }
        }
        else {
            if ($ne > 1) {
                $outtext .= "${ctype} (*${var})[$ne]";
            } else {
                $outtext .= "$ctype *$var";
            }
        }

        return $outtext;
    };

    my $return = sub {
        my ($args)   = @_;
        my $outtext  = $args->{outtext};
        return $outtext;
    };

    my $outtext = pg2_foreach_ioport_of_type($io, $do, $return);

    return $outtext;
}


sub _generate_ioconv_prototype {
    my ($io) = @_;

    my $do = sub {
        my ($args)   = @_;
        my $var      = $args->{var};
        my $sigoff   = $args->{signaloff};
        my $index    = $args->{index};
        my $ctype    = $args->{ctype};
        my $type    = $args->{type};
        my $outtext = q{};

        return q{} if $index > 0;
        return q{} if $sigoff > 0;

        if ($io eq 'FOSET') {
            $outtext .= "static inline $ctype convert_$var(int devid, UINT64 src);\n";
        } else {
            $outtext .= "static inline UINT64 convert_$var(int devid, $ctype src);\n";
        }
        return $outtext;
    };

    my $return = sub {
        my ($args)   = @_;
        my $outtext  = $args->{outtext};
        return $outtext;
    };

    my $outtext = pg2_foreach_ioport_of_type($io, $do, $return);

    return $outtext;
}

sub _generate_iorange_prototype {
    my ($io) = @_;

    my $do = sub {
        my ($args)   = @_;
        my $var      = $args->{var};
        my $sigoff   = $args->{signaloff};
        my $index    = $args->{index};
        my $ctype    = $args->{ctype};
        my $type    = $args->{type};
        my $nvar     = $args->{nsignal};
        my $outtext = q{};

        return q{} if $index > 0;
        return q{} if $sigoff > 0;
        return q{} if $ctype ne 'double';

        my $indent = q{ } x 4;
        if ($io =~ /JPSET|IPSET|COEFFSET/ && $type =~ /int/) {
            $outtext .= $indent . "void ${Prefix}_set_range_${var}(double min, double max);\n";
            $outtext .= $indent . "void ${Prefix}_set_range_${var}MC(int devid, double min, double max);\n";
            $outtext .= $indent . "void ${Prefix}_get_range_${var}(double *min, double *max);\n";
            $outtext .= $indent . "void ${Prefix}_get_range_${var}MC(int devid, double *min, double *max);\n";
            $outtext .= $indent . "inline double ${Prefix}_get_scale_${var}(void);\n";
            $outtext .= $indent . "inline double ${Prefix}_get_scale_${var}MC(int devid);\n";
            $outtext .= $indent . "inline double ${Prefix}_get_offset_${var}(void);\n";
            $outtext .= $indent . "inline double ${Prefix}_get_offset_${var}MC(int devid);\n";
        }
        elsif (($io =~ /FOSET/) ||
               ($io =~ /JPSET|IPSET|COEFFSET/ && $type =~ /float|log/)) {
            $outtext .= $indent . "inline void   ${Prefix}_set_scale_${var}(double scale);\n";
            $outtext .= $indent . "inline void   ${Prefix}_set_scale_${var}MC(int devid, double scale);\n";
            $outtext .= $indent . "inline double ${Prefix}_get_scale_${var}(void);\n";
            $outtext .= $indent . "inline double ${Prefix}_get_scale_${var}MC(int devid);\n";
        }
        return $outtext;
    };

    my $return = sub {
        my ($args)   = @_;
        my $outtext  = $args->{outtext};
        return $outtext;
    };

    my $outtext = pg2_foreach_ioport_of_type($io, $do, $return);

    return $outtext;
}

sub _generate_ioconv_definition {
    my ($io) = @_;

    my $do = sub {
        my ($args)   = @_;
        my $var      = $args->{var};
        my $sigoff   = $args->{signaloff};
        my $index    = $args->{index};
        my $type     = $args->{type};
        my $w        = $args->{width};
        my $m        = $args->{mantissa};
        my $ctype    = $args->{ctype};
        my $offset   = $args->{offset};
        my $outtext = q{};
        return q{} if $index > 0;
        my $t = defined $m ? ($type . $w . q{.} . $m) : ($type . $w);
        my $var_uc = ucfirst $var;

        #### embedded fragments of C source codes ########
        my $preamble_in .= << "END_PREAMBLE_IN";
static inline UINT64
convert_$var(int devid, $ctype src)
{
    UINT64 dst;

    WARN(3, "convert '$var' from $ctype to $t.\\n");
END_PREAMBLE_IN
        ##################################################
        my $preamble_out .= << "END_PREAMBLE_OUT";
static inline $ctype
convert_$var(int devid, UINT64 src)
{
    double dst;

    WARN(3, "convert '$var' from $t to $ctype.\\n");
END_PREAMBLE_OUT
        ##################################################
        my $postamble = << "END_POSTAMBLE";
    return dst;
}

END_POSTAMBLE
        ##################################################
        my $body_in1 = << "END_BODY_IN1";
    double scale  = ${Prefix}_get_scale_${var}MC(devid);
    double offset = ${Prefix}_get_offset_${var}MC(devid);
    dst = (UINT64)((src - offset) * scale + ONEHALF);

END_BODY_IN1
        #### end of C source codes #######################

        return q{} if $sigoff > 0;

        my $bodytext = q{};
        my $indent = q{ } x 4;
        if ($io eq 'FOSET') {
            $outtext .= $preamble_out;
	    $bodytext .= $indent;
	    $bodytext .= qq{// fprintf(stdout, "${var}:%llx\\n", src);\n};
            if ($ctype eq 'double') {
                for ($type) {
                    /int/ and do {
                        if ($w == 32 or $w == 64) {
                            $bodytext .= $indent;
                            $bodytext .= "dst = ${var_uc}_scale[devid] * (INT$w)src;\n";
                        }
                        elsif ($w <= 64) {
                            croak "fout of type int can have 32 or 64-bit width only\n" .
                              "(arbitrary width will be supported soon).\n";
                        }
                        else {
                            croak "too wide width ($w). fout of type int can have 64-bit width at maximum.\n";
                        }
                        last;
                    };

                    /float/ and do {
                        $bodytext .= $indent;
                        $bodytext .= "pg_conv_float_to_cdouble(src, $w, $m, &dst);\n";
                        $bodytext .= $indent;
                        $bodytext .= "dst *= ${var_uc}_scale[devid];\n";
                        last;
                    };

                    croak "currently fout of type $type is not supported.\n";
                }
            }
        }
        else {
            $outtext .= $preamble_in;
            for ($type) {
                /int/ and do {
                    if ($ctype eq 'double') {
                        $bodytext .= $body_in1;
                    } elsif ($ctype eq 'int') {
                        $bodytext .= $indent;
                        $bodytext .= "dst = (UINT64)src;\n";
                    }
                    last;
                };
                /(float|log)/ and do {
                    $bodytext .= $indent;
                    $bodytext .= "pg_conv_c${ctype}_to_${type}(src * ${var_uc}_scale[devid], &dst, $w, $m);\n";
                    last;
                };
            }
	    $bodytext .= $indent;
	    $bodytext .= qq{// fprintf(stdout, "${var}:%llx\\n", dst);\n};
        }
        if ($bodytext eq q{}) {
            $bodytext .= $indent;
            $bodytext .= qq/fprintf(stderr, "conversion from '${ctype}' to '${type}' is not supported. abort.\\n");\n/;
            $bodytext .= $indent;
            $bodytext .= "exit(1);\n\n";
        }
        ;
        $outtext .= $bodytext;
        $outtext .= $postamble;
        return $outtext;
    };

    my $return = sub {
        my ($args)   = @_;
        my $outtext  = $args->{outtext};
        return $outtext;
    };

    my $outtext = pg2_foreach_ioport_of_type($io, $do, $return);

    return $outtext;
}

sub _remove_unused_func {
    my ($intext) = @_;
    my $outtext = $intext;

    for my $ioname ('COEFFSET', 'JPSET', 'IPSET', 'FOSET') {
        my $params = pg2_get_parameter($ioname);
        my $found = defined $params ? 1 : 0;
        my $tagname = 'FOUND_' . $ioname;

        if ($found) {
            $outtext =~ s{<$tagname>\s*?\n(.*?)</$tagname>\s*?\n}{$1}xmsg;
        }
        else {
            $outtext =~ s{<$tagname>\s*?\n(.*?)</$tagname>\s*?\n}{}xmsg;
        }
    }
    return $outtext;
}

1;

__END__
