/*
    PCI Express Host Interface Bridge (EHIB) device driver for GRAPE-7

    Copyright (C) 2007 Atsushi Kawai

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/



#include <asm/current.h>

#if defined(CONFIG_SMP) && !defined(__SMP__)
#define __SMP__
#endif

#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <linux/version.h>
#include <asm/io.h>
#include <asm/page.h>
#include <asm/pgtable.h>
#include <asm/uaccess.h>
#include "grape7e.h"

#if 0
#define Cprintk(fmt, args...) printk(fmt, ## args)
#else
#define Cprintk(fmt, args...)
#endif

#ifndef LINUX_VERSION_CODE
#define LINUX_VERSION_CODE 0x020600
#warning LINUX_VERSION_CODE undefined. assume version 0x020600.
#endif

#if LINUX_VERSION_CODE < 0x020400
#warning "older than kernel 2.4"
#elif LINUX_VERSION_CODE < 0x020600
#warning "older than kernel 2.6 but newer than 2.3"
#else 
#warning "kernel 2.6 or later"
#endif

/* kernel 2.2 or earlier */
#if LINUX_VERSION_CODE < 0x020400
#define pci_resource_start(dev, region) (dev->base_address[region])
#ifndef VMA_OFFSET
#define VMA_OFFSET(vma) (vma->vm_offset)
#endif
#ifndef VM_RESERVED
#define VM_RESERVED (0)
#endif
#define Page unsigned long
int
pci_enable_device(dev)
{
    return (0);
}
static Page
virt_to_page(unsigned long virtual)
{
    return virtual;
}
static void
get_page(Page pageptr)
{
    atomic_inc(&mem_map[MAP_NR(pageptr)].count);
}

#else /* kernel 2.4 or later */

#ifndef VMA_OFFSET
#define VMA_OFFSET(vma) ((vma->vm_pgoff)<<PAGE_SHIFT)
#endif
#define Page struct page*
#endif /* LINUX_VERSION_CODE */

/* file operation methods */
static int grape7e_open(struct inode *ip, struct file *fp);
static int grape7e_close(struct inode *ip, struct file *fp);
static int n_grape7e_exist(int vendorid, int deviceid);
static int grape7e_ioctl(struct inode *ip, struct file *fp,
			unsigned int cmd, unsigned long arg);
static int grape7e_mmap(struct file *fp, struct vm_area_struct *vp);

/* vm operation methods */
static Page grape7e_dmar_vma_nopage(struct vm_area_struct *vp,
				    unsigned long address, int *type);
static Page grape7e_dmaw_vma_nopage(struct vm_area_struct *vp,
				    unsigned long address, int *type);
static void grape7e_vma_open(struct vm_area_struct *vp);
static void grape7e_vma_close(struct vm_area_struct *vp);

static void release_pages(int order, struct pci_dev *dev, unsigned long *va, dma_addr_t pa);

/* local defs */
static int major;
static struct pci_dev *grape7e[NGRAPE7E];
static unsigned long *dmarbuf[NGRAPE7E];
static unsigned long *dmawbuf[NGRAPE7E];
static dma_addr_t dmarbufpa[NGRAPE7E];
static dma_addr_t dmawbufpa[NGRAPE7E];
static unsigned int mapmode[NGRAPE7E];
static int ngrape7e; /* # of grape7e devices found */
static struct file_operations fops =
{
    .ioctl = grape7e_ioctl,
    .mmap = grape7e_mmap,
    .open = grape7e_open,
    .release = grape7e_close,
};
static struct vm_operations_struct dmar_vmops =
{
    .open = grape7e_vma_open,
    .close = grape7e_vma_close,
    .nopage = grape7e_dmar_vma_nopage,
};

static struct vm_operations_struct dmaw_vmops =
{
    .open = grape7e_vma_open,
    .close = grape7e_vma_close,
    .nopage = grape7e_dmaw_vma_nopage,
};

static void
release_pages(int order, struct pci_dev *dev,
	   unsigned long *va, dma_addr_t pa)
{
  int i;
  Page toppg = virt_to_page((unsigned long)va);

  for (i = 1; i < (1 << order); i++) { /* note that i starts from 1, not from 0.
					  reference count of the 1st page is automatically
					  handled by pci_free/alloc_consistent. */
      put_page_testzero(toppg + i);
  }
  pci_free_consistent(dev, PAGE_SIZE * (1<<order), va, pa);
}

int
grape7e_init(void)
{
    int i, ii;
    Page pageptr;

    /* register this device driver */
    major = register_chrdev(0, GRAPE7E_DEVNAME, &fops);

    /* init static vars */
    for (i = 0; i < NGRAPE7E; i++) {
	grape7e[i] = NULL;
	dmarbuf[i] = NULL;
	dmawbuf[i] = NULL;
	mapmode[i] = GRAPE7E_MAP_HIBMEM;
    }

    /* look for grape7e device(s) */
    ngrape7e = n_grape7e_exist(0x1556, 0x0e70);
    if (!ngrape7e) {
        unregister_chrdev(major, GRAPE7E_DEVNAME); 
	return (-ENODEV);
    }
    Cprintk(KERN_ALERT "bus: %d func: %d bar2 %08lx\n",
	   grape7e[0]->bus->number,
	   grape7e[0]->devfn,
	   grape7e[0]->base_address[2]);

    /* DMA configuration */
    for (i = 0; i < ngrape7e; i++) {
	/* set DMA masks */
	if (pci_set_dma_mask(grape7e[i], 0xffffffff)) {
	    printk(KERN_WARNING "grape7e: No suitable DMA available.\n");
            unregister_chrdev(major, GRAPE7E_DEVNAME); 
	    return (-ENODEV);
	}
	// printk(KERN_WARNING "grape7e: DMA mask set to 32-bit.\n");

	if (pci_set_consistent_dma_mask(grape7e[i], 0xffffffff)) {
	    printk(KERN_WARNING "grape7e: No suitable consistent DMA available.\n");
            unregister_chrdev(major, GRAPE7E_DEVNAME); 
	    return (-ENODEV);
	}
	// printk(KERN_WARNING "grape7e: consistent DMA mask set to 32-bit.\n");

	/* allocate DMA read buffers */
	dmarbuf[i] = pci_alloc_consistent(grape7e[i], GRAPE7E_DMABUF_BYTES, &dmarbufpa[i]);
	if (NULL == dmarbuf[i]) {
	    printk(KERN_ALERT "grape7e: pci_alloc_consistent failed\n");
	    for (ii = i-1; ii >= 0; ii--) {
	        release_pages(GRAPE7E_DMABUF_ORDER, grape7e[ii], dmarbuf[ii], dmarbufpa[ii]);
	    }
	    for (ii = i-1; ii >= 0; ii--) {
	        release_pages(GRAPE7E_DMABUF_ORDER, grape7e[ii], dmawbuf[ii], dmawbufpa[ii]);
	    }
            unregister_chrdev(major, GRAPE7E_DEVNAME); 
	    return (-ENOMEM);
	}
	pageptr = virt_to_page((unsigned long)dmarbuf[i]);
	for (ii = 1; ii < (1 << GRAPE7E_DMABUF_ORDER); ii++) {
	  get_page(pageptr + ii);
	}

	/* allocate DMA write buffers */
	dmawbuf[i] = pci_alloc_consistent(grape7e[i], GRAPE7E_DMABUF_BYTES, &dmawbufpa[i]);
	if (NULL == dmawbuf[i]) {
	    printk(KERN_ALERT "grape7e: pci_alloc_consistent failed\n");
	    for (ii = i; ii >= 0; ii--) {
	        release_pages(GRAPE7E_DMABUF_ORDER, grape7e[ii], dmarbuf[ii], dmarbufpa[ii]);
	    }
	    for (ii = i-1; ii >= 0; ii--) {
	        release_pages(GRAPE7E_DMABUF_ORDER, grape7e[ii], dmawbuf[ii], dmawbufpa[ii]);
	    }
            unregister_chrdev(major, GRAPE7E_DEVNAME); 
	    return (-ENOMEM);
	}
	pageptr = virt_to_page((unsigned long)dmawbuf[i]);
	for (ii = 1; ii < (1 << GRAPE7E_DMABUF_ORDER); ii++) {
	  get_page(pageptr + ii);
	}
    }

    // just for test
    for (i = 0; i < 4097; i++) {
      dmawbuf[0][i] = 0x12345678abc00000ll | i;
    }

    printk(KERN_ALERT "grape7e installed\n");

    return (0);
}

void
grape7e_exit(void)
{
    int i;

    for (i = 0; i < ngrape7e; i++) {
        release_pages(GRAPE7E_DMABUF_ORDER, grape7e[i], dmarbuf[i], dmarbufpa[i]);
        release_pages(GRAPE7E_DMABUF_ORDER, grape7e[i], dmawbuf[i], dmawbufpa[i]);
    }

    unregister_chrdev(major, GRAPE7E_DEVNAME);
    printk(KERN_ALERT "grape7e uninstalled\n");
}

module_init(grape7e_init);
module_exit(grape7e_exit);
MODULE_LICENSE("Dual BSD/GPL");

static int
grape7e_open(struct inode *ip, struct file *fp)
{
    Cprintk(KERN_ALERT "grape7e_open\n");
    /*
    MOD_INC_USE_COUNT;
    */
    return (0);
}

static int
grape7e_close(struct inode *ip, struct file *fp)
{
    Cprintk(KERN_ALERT "grape7e_close\n");
    /*
    MOD_DEC_USE_COUNT;
    */
    return (0);
}

static int
grape7e_ioctl(struct inode *ip, struct file *fp, 
	     unsigned int cmd, unsigned long arg)
{
    int minor = MINOR(ip->i_rdev);
    Cprintk(KERN_ALERT "grape7e_ioctl cmd: %x\n", (int) cmd);

    /* exec cmd with argument arg */
    switch (cmd) {
      case GRAPE7E_WRITE_CFG: {
          struct long_access myarg;
	  copy_from_user(&myarg, (struct long_access *)arg, sizeof(myarg));
	  pci_write_config_dword(grape7e[minor], myarg.addr, myarg.data);

	  Cprintk(KERN_ALERT "grape7e_ioctl done WRITE_CFG\n");

	  break;
      }
      case GRAPE7E_READ_CFG: {
	  struct long_access myarg;

	  Cprintk(KERN_ALERT "grape7e_ioctl will READ_CFG\n");
	  copy_from_user(&myarg, (struct long_access *)arg, sizeof(myarg));

	  pci_read_config_dword(grape7e[minor], myarg.addr, &(myarg.data));
	  Cprintk(KERN_ALERT "grape7e_ioctl READ_CFG  addr: %x  data: %x\n", myarg.addr, myarg.data);

	  copy_to_user((struct long_access *)arg, &myarg, sizeof(myarg));

	  Cprintk(KERN_ALERT "grape7e_ioctl done READ_CFG\n");

	  break;

      }
      case GRAPE7E_GET_DMAR_PA: {
	  unsigned long baddr;

	  Cprintk(KERN_ALERT "grape7e_ioctl will GET_DMAR_PA\n");

	  baddr = dmarbufpa[minor];
	  copy_to_user((unsigned int *)arg, &baddr, sizeof(baddr));

	  Cprintk(KERN_ALERT "GET_DMAR_PA: DMA read buf bus addr: %016lx\n", baddr);
	  Cprintk(KERN_ALERT "grape7e_ioctl done GET_DMAR_PA\n");
	  break;
      }
      case GRAPE7E_GET_DMAW_PA: {
	  unsigned long baddr;

	  Cprintk(KERN_ALERT "grape7e_ioctl will GET_DMAW_PA\n");

	  baddr = dmawbufpa[minor];
	  copy_to_user((unsigned int *)arg, &baddr, sizeof(baddr));

	  Cprintk(KERN_ALERT "GET_DMAW_PA: DMA write buf bus addr: %016lx\n", baddr);
	  Cprintk(KERN_ALERT "grape7e_ioctl done GET_DMAW_PA\n");
	  break;
      }
      case GRAPE7E_SET_MAP_MODE: {
          Cprintk(KERN_ALERT "grape7e_ioctl will SET_MAP_MODE\n");

	  mapmode[minor] = (unsigned int)arg;
	  Cprintk(KERN_ALERT "SET_MAP_MODE: mapmode: %d\n", mapmode[minor]);

	  Cprintk(KERN_ALERT "grape7e_ioctl done SET_MAP_MODE\n");

	  break;
      }
      default:
	return (-EINVAL);
    }
    return (0);
}

static Page
grape7e_dmar_vma_nopage(struct vm_area_struct *vp,
	      unsigned long address, int *type)
{
    int minor = MINOR(vp->vm_file->f_dentry->d_inode->i_rdev);
    Page pageptr;
    unsigned long offset = address - vp->vm_start + VMA_OFFSET(vp);

    Cprintk(KERN_ALERT "grape7e_dmar_vma_nopage minor: %d\n", minor);
    pageptr = virt_to_page((unsigned long)dmarbuf[minor] + offset);
    get_page(pageptr);

    return pageptr;
}

static Page
grape7e_dmaw_vma_nopage(struct vm_area_struct *vp,
	      unsigned long address, int *type)
{
    int minor = MINOR(vp->vm_file->f_dentry->d_inode->i_rdev);
    Page pageptr;
    unsigned long offset = address - vp->vm_start + VMA_OFFSET(vp);

    Cprintk(KERN_ALERT "grape7e_dmaw_vma_nopage minor: %d\n", minor);
    pageptr = virt_to_page((unsigned long)dmawbuf[minor] + offset);
    get_page(pageptr);

    Cprintk(KERN_ALERT "pageptr: 0x%08x  dmawbuf[%d]: 0x%08x  offset: 0x%08x\n",
	   pageptr, minor, dmawbuf[minor], offset);

    return pageptr;
}

static void
grape7e_vma_open(struct vm_area_struct *vp)
{
    Cprintk(KERN_ALERT "grape7e_vma_open\n");
    // MOD_INC_USE_COUNT;
}

static void
grape7e_vma_close(struct vm_area_struct *vp)
{
    Cprintk(KERN_ALERT "grape7e_vma_close\n");
    // MOD_DEC_USE_COUNT;
}

static int
grape7e_mmap(struct file *fp, struct vm_area_struct *vp)
{
    unsigned long off = VMA_OFFSET(vp);
    unsigned long virtual = vp->vm_start;
    unsigned long physical;
    unsigned long vsize = vp->vm_end - vp->vm_start;
    int ret = 0;
    int minor = MINOR(fp->f_dentry->d_inode->i_rdev);
    int bar;

    switch (mapmode[minor]) {
      case GRAPE7E_MAP_HIBMEM: /* map XHIB local memory */
	bar = 0;
	break;
      case GRAPE7E_MAP_BACKEND: /* map XHIB backend address space */
	bar = 1;
	break;
      case GRAPE7E_MAP_DBLBUF: /* map XHIB internal double SRAM buffers */
	bar = 2;
	break;
      default:
	bar = 0;
    }

    switch (mapmode[minor]) {
      case GRAPE7E_MAP_HIBMEM: /* map XHIB local memory */
      case GRAPE7E_MAP_BACKEND: /* map XHIB backend address space */
      case GRAPE7E_MAP_DBLBUF: /* map XHIB internal double SRAM buffers */

	physical  = pci_resource_start(grape7e[minor], bar) & PCI_BASE_ADDRESS_MEM_MASK;
	/* map BAR to the user space */
	/* printk(KERN_ALERT "bar%d physical: 0x%0x\n", bar, physical); */
	physical += off;
	// vp->vm_page_prot = pgprot_noncached(vp->vm_page_prot);
	vp->vm_flags |= VM_RESERVED;

#if LINUX_VERSION_CODE < 0x020503 /* earlier than 2.5.3 */
	ret = remap_page_range(virtual, physical, vsize, vp->vm_page_prot);
#elif LINUX_VERSION_CODE < 0x02060a /* earlier than 2.6.10 */
	ret = remap_page_range(vp, virtual, physical, vsize, vp->vm_page_prot);
#else /* 2.6.10 or later */
      ret = remap_pfn_range(vp, virtual, physical >> PAGE_SHIFT, vsize, vp->vm_page_prot);
#endif

	if (ret) return (-EAGAIN);
	/* printk(KERN_ALERT "GRAPE7E_MAP_HIBMEM done\n"); */
	break;

    case GRAPE7E_MAP_DMARBUF: /* map DMA read buf */
	vp->vm_ops = &dmar_vmops;
	// vp->vm_page_prot = pgprot_noncached(vp->vm_page_prot); // !!!  may be dangerous to remove this line.
	vp->vm_flags |= VM_RESERVED | VM_IO;
	grape7e_vma_open(vp);
	Cprintk(KERN_ALERT "GRAPE7E_MAP_DMARBUF done\n");
	break;

    case GRAPE7E_MAP_DMAWBUF: /* map DMA write buf */
	vp->vm_ops = &dmaw_vmops;
        // vp->vm_page_prot = pgprot_noncached(vp->vm_page_prot); // !!!
	vp->vm_flags |= VM_RESERVED | VM_IO;
	grape7e_vma_open(vp);
	Cprintk(KERN_ALERT "GRAPE7E_MAP_DMAWBUF done\n");
	break;

    default:
	printk(KERN_ALERT "unknown map mode %d\n", mapmode[minor]);
	ret = 1;
	break;
    }
    if (ret) {
	return (-EAGAIN);
    }
    return (ret);
}

static int
n_grape7e_exist(int vendorid, int deviceid)
{
    int i;
    struct pci_dev *grape7e0 = NULL;

    for (i = 0; ; i++) {
	grape7e[i] = pci_find_device(vendorid, deviceid, grape7e0);
	grape7e0 = grape7e[i];
	if (!grape7e[i]) {
	    break;
	}
	else if (pci_enable_device(grape7e[i])) {
	    break;
	}
    }
    if (i == 0) {
	printk(KERN_ALERT "no grape7e found\n");
	return (-ENODEV);
    }
    else {
	printk(KERN_ALERT "%d grape7e(s) found\n", i);
	return (i);
    }
}
