#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include "hibutil.h"

/* HIB local definition(s) & variable(s) */
static Hib hib[NHIB];
static int hib_fd[NHIB];
static unsigned long dmar_pa[NHIB]; /* physical addres of hib[].dmar_buf */
static unsigned long dmaw_pa[NHIB]; /* physical addres of hib[].dmaw_buf */
static void (*SendMCfunc[NHIB])(int devid, int size, UINT64 *buf);
static int (*RecvMCfunc[NHIB])(int devid, int size, UINT64 *buf);

/* HIB local function(s) */
static void init_envs(int devid);
static void lock_hib(int devid);
static void unlock_hib(int devid);
static void TBmemMap(int devid);
static void TBgetDmaInfo(int devid,
			 unsigned long *dmarpa, UINT64 **dmarva,
			 unsigned long *dmawpa, UINT64 **dmawva);

#define warn(lv, fmt, args...) if (lv <= warn_level) fprintf(stderr, fmt, ## args);
static int warn_level = 2; /* warning message output level. the higher the more verbose.
                              0: no warning (may cause wrong result with g7pkg/scripts/check.csh)
                              1: minimum
                              2: default
                              3: for debugging purpose
                           */

void
hib_set_warn_level(int lv)
{
    warn_level = lv;
}

int
hib_recvMC(int devid, int size, UINT64 *buf)
{
    return RecvMCfunc[devid](devid, size, buf);
}

/*
 * open HIB
 * returns a ponter to the Hib object opened.
 */
Hib*
hib_openMC(int devid)
{
    init_envs(devid);

    hib_fd[devid] = open(hib[devid].devname, O_RDWR);
    if (hib_fd[devid] < 0) {
	perror(hib[devid].devname);
	exit(2);
    }

    lock_hib(devid);

    /*
     * set local register addresses
     */
    switch(hib[devid].type) {

      case HIB_GRAPE7X:
        hib[devid].dmabuf_bytes = GRAPE7X_DMABUF_BYTES;
        hib[devid].piowbuf_bytes = GRAPE7X_PIOWBUF_BYTES;
        hib[devid].local_mem_bytes = GRAPE7X_LOCAL_MEM_BYTES;
        hib[devid].backend_mem_bytes = GRAPE7X_BACKEND_MEM_BYTES;

        hib[devid].write_cfg = GRAPE7X_WRITE_CFG;
        hib[devid].read_cfg = GRAPE7X_READ_CFG;
        hib[devid].get_dmaw_pa = GRAPE7X_GET_DMAW_PA;
        hib[devid].get_dmar_pa = GRAPE7X_GET_DMAR_PA;
        hib[devid].set_map_mode = GRAPE7X_SET_MAP_MODE;

        hib[devid].map_hibmem = GRAPE7X_MAP_HIBMEM;
        hib[devid].map_dmarbuf = GRAPE7X_MAP_DMARBUF;
        hib[devid].map_dmawbuf = GRAPE7X_MAP_DMAWBUF;
        hib[devid].map_backend = GRAPE7X_MAP_BACKEND;
        hib[devid].map_dblbuf = GRAPE7X_MAP_DBLBUF;

        hib[devid].boardinfo = GRAPE7X_BOARDINFO;
        hib[devid].dma0lsb = GRAPE7X_DMA0LSB;
        hib[devid].dma0msb = GRAPE7X_DMA0MSB;
        hib[devid].dma0size = GRAPE7X_DMA0SIZE;
        hib[devid].dma0cmd = GRAPE7X_DMA0CMD;
        hib[devid].dma0misc = GRAPE7X_DMAMISC;
        hib[devid].dma1lsb = GRAPE7X_DMA1LSB;
        hib[devid].dma1msb = GRAPE7X_DMA1MSB;
        hib[devid].dma1size = GRAPE7X_DMA1SIZE;
        hib[devid].dma1cmd = GRAPE7X_DMA1CMD;
        hib[devid].dma1misc = GRAPE7X_DMAMISC;
        hib[devid].dmastat = GRAPE7X_DMASTAT;
        hib[devid].mailbox = GRAPE7X_MAILBOX;
        hib[devid].asmi = GRAPE7X_ASMI;
        hib[devid].pllconf = GRAPE7X_PLLCONF;

        hib[devid].dma0cmd_kickoff           = GRAPE7X_DMA0CMD_KICKOFF;
        hib[devid].dma0misc_done_bit         = GRAPE7X_DMA0MISC_DONE_BIT;
        hib[devid].dma0misc_swap_sram_bit    = GRAPE7X_DMA0MISC_SWAP_SRAM_BIT;
        hib[devid].dma0misc_sram_wcnt_bit    = GRAPE7X_DMA0MISC_SRAM_WCNT_BIT;
        hib[devid].dma1cmd_kickoff           = GRAPE7X_DMA1CMD_KICKOFF;
        hib[devid].dma1misc_done_bit         = GRAPE7X_DMA1MISC_DONE_BIT;
        hib[devid].dmastat_reset_backend_bit = GRAPE7X_DMASTAT_RESET_BACKEND_BIT;
        hib[devid].dmastat_dma_reset_bit     = GRAPE7X_DMASTAT_DMA_RESET_BIT;
        hib[devid].pllconf_write_bit         = GRAPE7X_PLLCONF_WRITE_BIT;
        hib[devid].pllconf_reconfig_bit      = GRAPE7X_PLLCONF_RECONFIG_BIT;
        hib[devid].pllconf_busy_bit          = GRAPE7X_PLLCONF_BUSY_BIT;

        hib[devid].asmi_error_bit = GRAPE7X_ASMI_ERROR_BIT;
        hib[devid].asmi_busy_bit  = GRAPE7X_ASMI_BUSY_BIT;
        hib[devid].asmi_cmd_rsid  = GRAPE7X_ASMI_CMD_RSID;
        hib[devid].asmi_cmd_wp    = GRAPE7X_ASMI_CMD_WP;
        hib[devid].asmi_cmd_eb    = GRAPE7X_ASMI_CMD_EB;

        break;

      case HIB_GRAPE7E:
        hib[devid].dmabuf_bytes = GRAPE7E_DMABUF_BYTES;
        hib[devid].piowbuf_bytes = GRAPE7E_PIOWBUF_BYTES;
        hib[devid].local_mem_bytes = GRAPE7E_LOCAL_MEM_BYTES;
        hib[devid].backend_mem_bytes = GRAPE7E_BACKEND_MEM_BYTES;

        hib[devid].write_cfg = GRAPE7E_WRITE_CFG;
        hib[devid].read_cfg = GRAPE7E_READ_CFG;
        hib[devid].get_dmaw_pa = GRAPE7E_GET_DMAW_PA;
        hib[devid].get_dmar_pa = GRAPE7E_GET_DMAR_PA;
        hib[devid].set_map_mode = GRAPE7E_SET_MAP_MODE;

        hib[devid].map_hibmem = GRAPE7E_MAP_HIBMEM;
        hib[devid].map_dmarbuf = GRAPE7E_MAP_DMARBUF;
        hib[devid].map_dmawbuf = GRAPE7E_MAP_DMAWBUF;
        hib[devid].map_backend = GRAPE7E_MAP_BACKEND;
        hib[devid].map_dblbuf = GRAPE7E_MAP_DBLBUF;

        hib[devid].boardinfo = GRAPE7E_BOARDINFO;
        hib[devid].dma0lsb = GRAPE7E_DMA0LSB;
        hib[devid].dma0msb = GRAPE7E_DMA0MSB;
        hib[devid].dma0size = GRAPE7E_DMA0SIZE;
        hib[devid].dma0cmd = GRAPE7E_DMA0CMD;
        hib[devid].dma0misc = GRAPE7E_DMA0MISC;
        hib[devid].dma1lsb = GRAPE7E_DMA1LSB;
        hib[devid].dma1msb = GRAPE7E_DMA1MSB;
        hib[devid].dma1size = GRAPE7E_DMA1SIZE;
        hib[devid].dma1cmd = GRAPE7E_DMA1CMD;
        hib[devid].dma1misc = GRAPE7E_DMA1MISC;
        hib[devid].dmastat = GRAPE7E_DMASTAT;
        hib[devid].mailbox = GRAPE7E_MAILBOX;
        hib[devid].asmi = GRAPE7E_ASMI;
        hib[devid].pllconf = GRAPE7E_PLLCONF;

        hib[devid].dma0cmd_kickoff           = GRAPE7E_DMA0CMD_KICKOFF;
        hib[devid].dma0misc_done_bit         = GRAPE7E_DMA0MISC_DONE_BIT;
        hib[devid].dma0misc_swap_sram_bit    = GRAPE7E_DMA0MISC_SWAP_SRAM_BIT;
        hib[devid].dma0misc_sram_wcnt_bit    = GRAPE7E_DMA0MISC_SRAM_WCNT_BIT;
        hib[devid].dma1cmd_kickoff           = GRAPE7E_DMA1CMD_KICKOFF;
        hib[devid].dma1misc_done_bit         = GRAPE7E_DMA1MISC_DONE_BIT;
        hib[devid].dmastat_reset_backend_bit = GRAPE7E_DMASTAT_RESET_BACKEND_BIT;
        hib[devid].dmastat_dma_reset_bit     = GRAPE7E_DMASTAT_DMA_RESET_BIT;
        hib[devid].pllconf_write_bit         = GRAPE7E_PLLCONF_WRITE_BIT;
        hib[devid].pllconf_reconfig_bit      = GRAPE7E_PLLCONF_RECONFIG_BIT;
        hib[devid].pllconf_busy_bit          = GRAPE7E_PLLCONF_BUSY_BIT;

        hib[devid].asmi_error_bit = GRAPE7E_ASMI_ERROR_BIT;
        hib[devid].asmi_busy_bit  = GRAPE7E_ASMI_BUSY_BIT;
        hib[devid].asmi_cmd_rsid  = GRAPE7E_ASMI_CMD_RSID;
        hib[devid].asmi_cmd_wp    = GRAPE7E_ASMI_CMD_WP;
        hib[devid].asmi_cmd_eb    = GRAPE7E_ASMI_CMD_EB;

        break;

      default:
        fprintf(stderr, "hib_openMC: unknown HIB type.\n", hib[devid].type);
        exit(1);
    }

    TBmemMap(devid);

    /*
     * maps DMA buffers
     */ 
    TBgetDmaInfo(devid,
		 &(dmar_pa[devid]), &(hib[devid].dmar_buf),
		 &(dmaw_pa[devid]), &(hib[devid].dmaw_buf));
    /*
     * dmar/w_buf: virtual address of DMA read/write buffer
     * dmar/w_pa: its physical address
     */
    warn(3, "hib[%d].dmar_buf: 0x%016lx pa: %016lx\n",
         devid, hib[devid].dmar_buf, dmar_pa[devid]);
    warn(3, "hib[%d].dmaw_buf: 0x%016lx pa: %016lx\n",
         devid, hib[devid].dmaw_buf, dmaw_pa[devid]);

    /* initialize local registers */
    hib_mem_writeMC(devid, hib[devid].dma0lsb, dmar_pa[devid]); /* LSB of DMA0 start address in PCI space */
    hib_mem_writeMC(devid, hib[devid].dma1lsb, dmaw_pa[devid]); /* LSB of DMA1 start address in PCI space */
    hib_mem_writeMC(devid, hib[devid].dma0msb, 0x0);
    hib_mem_writeMC(devid, hib[devid].dma1msb, 0x0);

    hib_set_test_modeMC(devid, TESTMODE_NONE);
    SendMCfunc[devid] = hib_dmarMC;
    RecvMCfunc[devid] = hib_dmawMC;

    return &(hib[devid]);
}

/* close HIB */
void
hib_closeMC(int devid)
{
    size_t psz = getpagesize();

    unlock_hib(devid);

    munmap((void *)hib[devid].local_reg,   (hib[devid].local_mem_bytes/psz+1)*psz);
    munmap((void *)hib[devid].backend,     (hib[devid].backend_mem_bytes/psz+1)*psz);
    munmap((void *)hib[devid].piow_dblbuf, (hib[devid].piowbuf_bytes/psz+1)*psz);
    munmap((void *)hib[devid].dmar_buf,    ((hib[devid].dmabuf_bytes-1)/psz+1)*psz);
    munmap((void *)hib[devid].dmaw_buf,    ((hib[devid].dmabuf_bytes-1)/psz+1)*psz);

    close(hib_fd[devid]);
    hib_fd[devid] = -1;
}

void
hib_set_test_modeMC(int devid, int mode)
{
    hib_mem_writeMC(devid, hib[devid].mailbox, mode);
}

void
hib_set_sendfuncMC(int devid, int func)
{
    switch (func) {
      case SENDFUNC_PIOW:
	  SendMCfunc[devid] = hib_piowMC;
	  break;
      case SENDFUNC_DMAR:
	  SendMCfunc[devid] = hib_dmarMC;
	  break;
      default:
	  break;
    }
}

void
hib_sendMC(int devid, int size, UINT64 *buf) /* size: transfer size in 8-byte word */
{
  SendMCfunc[devid](devid, size, buf);
}

void
hib_piowMC(int devid, int size, UINT64 *buf) /* size: transfer size in 8-byte word */
{
    int i, j;
    int s = (hib[devid].piowbuf_bytes>>3)-1;

    for (i = 0; i < size; i += s) {

      if (i + s > size) {
	s = size - i;
      }
      for (j = 0; j < s; j++) {
          hib[devid].piow_dblbuf[j] = buf[i + j];
      }
      hib_mem_writeMC(devid, hib[devid].dma0misc,
                      (1<<(hib[devid].dma0misc_swap_sram_bit)) |
                      (s<<(hib[devid].dma0misc_sram_wcnt_bit)));

      //      _mm_mfence();
      //      usleep(100000);
      // write '1' to HIB local register 'swap_sram'
      // write data size to 'sram_wcnt'
    }
}

/* DMA read (host -> pcixsys) using DMA0 channel */
void
hib_dmarMC(int devid, int size, UINT64 *buf) /* size: transfer size in 8-byte word */
{
    volatile unsigned long misc;
    unsigned long val;

    if (size > (hib[devid].dmabuf_bytes>>2)) {
	fprintf(stderr, "hib_dmarMC(): size too large.\n");
	fprintf(stderr, "size: %d\n", size);
	exit(2);
    }

    /*
     * DMA read fails on new core (ver 7.0.5) for unknown reason, if
     * data size is smaller than 16 bytes. That's why the below is
     * necessary.
     */
    // !!!
    if (size < 8) {
	int ii;
	for (ii = size; ii < 8; ii++) {
	    buf[ii] = 0; // fill buf with dummy data
	}
	size = 8;
    }

    val = dmar_pa[devid];
    hib_mem_writeMC(devid, hib[devid].dma0lsb, val);
    hib_mem_writeMC(devid, hib[devid].dma0size, size*8); /* size in byte */
    hib_mem_writeMC(devid, hib[devid].dma0cmd, hib[devid].dma0cmd_kickoff);

    //    __mm_memfence();

    /*
     * wait till the DMA transfer completes
     * dmadone(1:0) is mapped to DMADONE(1:0)
     */
    do {
	misc = hib_mem_readMC(devid, hib[devid].dma0misc);
    } while ((misc & (1<<hib[devid].dma0misc_done_bit)) == 0); /* wait until dmadone0 is asserted */
}

void
hib_start_dmarMC(int devid, int size, UINT64 *buf) /* size: transfer size in 8-byte word */
{
    unsigned long val;

    if (size > (hib[devid].dmabuf_bytes>>2)) {
	fprintf(stderr, "hib_dmarMC(): size too large.\n");
	fprintf(stderr, "size: %d\n", size);
	exit(2);
    }

    /*
     * DMA read fails on new core (ver 7.0.5) for unknown reason, if
     * data size is smaller than 16 bytes. That's why the below is
     * necessary.
     */
    // !!!
    if (size < 8) {
	int ii;
	for (ii = size; ii < 8; ii++) {
	    buf[ii] = 0; // fill buf with dummy data
	}
	size = 8;
    }

    val = dmar_pa[devid];
    hib_mem_writeMC(devid, hib[devid].dma0lsb, val);
    hib_mem_writeMC(devid, hib[devid].dma0size, size*8); /* size in byte */
    hib_mem_writeMC(devid, hib[devid].dma0cmd, hib[devid].dma0cmd_kickoff);
    //    __mm_memfence();
}

void
hib_finish_dmarMC(int devid)
{
    volatile unsigned long misc;

    /*
     * wait till the DMA transfer completes
     * dmadone(1:0) is mapped to DMADONE(1:0)
     */

    do {
	misc = hib_mem_readMC(devid, hib[devid].dma0misc);
    } while ((misc & (1<<hib[devid].dma0misc_done_bit)) == 0); /* wait until dmadone0 is asserted */
}

/* DMA write (host <- pcixsys) using DMA1 channel */
int
hib_dmawMC(int devid, int size, UINT64 *buf)
{
    volatile unsigned long misc;
    unsigned long val;
    int ret = 0; /* -1 on error */

    if (size > (hib[devid].dmabuf_bytes>>2)) {
        fprintf(stderr, "hib_dmawMC(): size too large.\n");
        fprintf(stderr, "size: %d\n", size);
        exit(2);
    }

    val = dmaw_pa[devid];
    hib_mem_writeMC(devid, hib[devid].dma1lsb, val);
    hib_mem_writeMC(devid, hib[devid].dma1msb, 0x0); /* start address MSB in local space */
    hib_mem_writeMC(devid, hib[devid].dma1size, size*8); /* size in byte */
    hib_mem_writeMC(devid, hib[devid].dma1cmd, hib[devid].dma1cmd_kickoff);
    /* command (mem write block, no scatter-gather, 64-bit, high priority)
     * 
     * 7-4: PCI transfer command
     *   3: scatter-gather
     *   2: 64-bit mode
     *   1: priority
     *   0: reserved
     */  

    /*
     * wait till the DMA transfer completes
     * dmadone(1:0) is mapped to DMADONE(1:0)
     */
    do {
        misc = hib_mem_readMC(devid, hib[devid].dma1misc);
    } while ((misc & (1<<hib[devid].dma1misc_done_bit)) == 0); /* wait until dmadone1 is asserted */

    return ret;
}

/* start DMA write (host <- pcixsys) using DMA1 channel */
void
hib_start_dmawMC(int devid, int size, UINT64 *buf)
{
    unsigned long val;

    if (size > (hib[devid].dmabuf_bytes>>2)) {
        fprintf(stderr, "hib_dmawMC(): size too large.\n");
        fprintf(stderr, "size: %d\n", size);
        exit(2);
    }

    val = dmaw_pa[devid];
    hib_mem_writeMC(devid, hib[devid].dma1lsb, val);
    hib_mem_writeMC(devid, hib[devid].dma1msb, 0x0); /* start address MSB in local space */
    hib_mem_writeMC(devid, hib[devid].dma1size, size*8); /* size in byte */
    hib_mem_writeMC(devid, hib[devid].dma1cmd, hib[devid].dma1cmd_kickoff);
    /* command (mem write block, no scatter-gather, 64-bit, high priority)
     * 
     * 7-4: PCI transfer command
     *   3: scatter-gather
     *   2: 64-bit mode
     *   1: priority
     *   0: reserved
     */  
}


/*
 * wait completion of the DMA write (host <- pcixsys) transaction
 * kicked off by hib_start_dmawMC()
 */
int
hib_finish_dmawMC(int devid)
{
    volatile unsigned long misc;
    int ret = 0; /* -1 on error */

    /*
     * wait till the DMA transfer completes
     * dmadone(1:0) is mapped to DMADONE(1:0)
     */
    do {
        misc = hib_mem_readMC(devid, hib[devid].dma1misc);
    } while ((misc & (1<<hib[devid].dma1misc_done_bit)) == 0); /* wait until dmadone1 is asserted */

    return ret;
}

UINT32
hib_config_readMC(int devid, UINT32 addr)
{
    struct long_access ca;
    ca.addr = addr;
    ca.data = 0;

    if (hib_fd[devid] < 0) {
	fprintf(stderr, "open hib[%d] first\n", devid);
	exit (1);
    }
    ioctl(hib_fd[devid], hib[devid].read_cfg, &ca);
    return (ca.data);
}

void
hib_config_writeMC(int devid, UINT32 addr, UINT32 value)
{
    struct long_access ca;
    ca.addr = addr;
    ca.data = value;

    if (hib_fd[devid] < 0) {
	fprintf(stderr, "open hib[%d] first\n", devid);
	exit (1);
    }
    ioctl(hib_fd[devid], hib[devid].write_cfg, &ca);
}

void
hib_mem_writeMC(int devid, UINT32 addr, UINT32 value)
{
    hib[devid].local_reg[addr>>2] = value;
}

UINT32
hib_mem_readMC(int devid, UINT32 addr)
{
    UINT32 ret;

    ret =  hib[devid].local_reg[addr>>2];
    return (ret);
}

void
TBgetDmaInfo(int devid,
	     unsigned long *dmarpa, UINT64 **dmarva,
	     unsigned long *dmawpa, UINT64 **dmawva)
     /*
      * pa: physical addr of DMA read/write buf alloced in the kernel space
      * va: virtual addr (in the user process context) pa is mapped to
      */
{
    size_t psz = getpagesize();

    /* map DMA read buffer */
    if (-1 == ioctl(hib_fd[devid], hib[devid].get_dmar_pa, dmarpa)) {
	perror("TBgetDmaInfo() failed");
	exit (1);
    };
    if (-1 == ioctl(hib_fd[devid], hib[devid].set_map_mode, GRAPE7X_MAP_DMARBUF)) {
	perror("TBgetDmaInfo() failed");
	exit (1);
    };
    *dmarva = mmap(NULL,
		   ((hib[devid].dmabuf_bytes-1)/psz+1)*psz,
		   PROT_READ | PROT_WRITE,
		   MAP_SHARED,
		   hib_fd[devid],
		   0);
    if ((long int)(*dmarva) == -1) {
	perror("TBgetDmaInfo");
	exit (1);
    }

    /* map DMA write buffer */
    if (-1 == ioctl(hib_fd[devid], hib[devid].get_dmaw_pa, dmawpa)) {
	perror("TBgetDmaInfo() failed");
	exit (1);
    };
    if (-1 == ioctl(hib_fd[devid], hib[devid].set_map_mode, GRAPE7X_MAP_DMAWBUF)) {
	perror("TBgetDmaInfo() failed");
	exit (1);
    };
    *dmawva = mmap(NULL,
		   ((hib[devid].dmabuf_bytes-1)/psz+1)*psz,
		   PROT_READ | PROT_WRITE,
		   MAP_SHARED,
		   hib_fd[devid],
		   0);
    if ((long int)(*dmawva) == -1) {
	perror("TBgetDmaInfo");
	exit (1);
    }
    if (-1 == ioctl(hib_fd[devid], hib[devid].set_map_mode, GRAPE7X_MAP_HIBMEM)) {
	perror("TBgetDmaInfo() failed");
	exit (1);
    };
}

static void
TBmemMap(int devid)
{
    size_t psz = getpagesize();

    /*
     * maps HIB local register (BAR0)
     */
    if (-1 == ioctl(hib_fd[devid], hib[devid].set_map_mode, GRAPE7X_MAP_HIBMEM)) {
	perror("TBgetDmaInfo() failed");
	exit (1);
    };
    hib[devid].local_reg = mmap(NULL,
                                 (hib[devid].local_mem_bytes/psz+1)*psz,
                                 PROT_READ | PROT_WRITE,
                                 MAP_SHARED,
                                 hib_fd[devid],
                                 0); /* device driver takes care of bar offset */
    if ((long int)(hib[devid].local_reg) == -1) {
	fprintf(stderr, "%s%d\n", hib[devid].devname, devid);
	exit (1);
    }
    warn(3, "hib[%d].local_reg: 0x%016lx\n", devid, hib[devid].local_reg);


    /*
     * maps BAR1 (HIB backend)
     */
    if (-1 == ioctl(hib_fd[devid], hib[devid].set_map_mode, GRAPE7X_MAP_BACKEND)) {
	perror("TBmemMap() failed");
	exit (1);
    };
    hib[devid].backend = mmap(NULL,
                              (hib[devid].backend_mem_bytes/psz+1)*psz,
                              PROT_READ | PROT_WRITE,
                              MAP_SHARED,
                              hib_fd[devid],
                              0); /* device driver takes care of bar offset */
    if ((long int)(hib[devid].backend) == -1) {
	fprintf(stderr, "%s%d\n", hib[devid].devname, devid);
	exit (1);
    }
    warn(3, "hib[%d].backend: 0x%016lx\n", devid, hib[devid].backend);


    /*
     * maps BAR2 (double buffer for PIO write transfer)
     */
    if (-1 == ioctl(hib_fd[devid], hib[devid].set_map_mode, GRAPE7X_MAP_DBLBUF)) {
	perror("TBmemMap() failed");
	exit (1);
    };
    hib[devid].piow_dblbuf = mmap(NULL,
                                   (hib[devid].piowbuf_bytes/psz+1)*psz,
                                   PROT_READ | PROT_WRITE,
                                   MAP_SHARED,
                                   hib_fd[devid],
                                   0); /* device driver takes care of bar offset */
    if ((long int)(hib[devid].backend) == -1) {
	fprintf(stderr, "%s%d\n", hib[devid].devname, devid);
	exit (1);
    }
    warn(3, "hib[%d].piow_dblbuf: 0x%016lx\n", devid, hib[devid].piow_dblbuf);


    /* house keeping */
    if (-1 == ioctl(hib_fd[devid], hib[devid].set_map_mode, GRAPE7X_MAP_HIBMEM)) {
	perror("TBgetDmaInfo() failed");
	exit (1);
    };
}

static void
init_envs(int devid)
{
    int i, j;
#if NHIB > 8
#error NHIB must not exceed 8
#endif
    static int firstcall[8] = {
        1, 1, 1, 1, 1, 1, 1, 1, 
    };

    if (firstcall[devid]) {
	firstcall[devid] = 0;
        hib_fd[devid] = -1;

	for (j = 0, i = 0; j < NGRAPE7X; j++, i++) {
            if (i == devid) {
                sprintf(hib[i].devname, "/dev/%s%d", GRAPE7X_DEVNAME, j);
                hib[i].type = HIB_GRAPE7X;
            }
	}
	for (j = 0; j < NGRAPE7E; j++, i++) {
            if (i == devid) {
                sprintf(hib[i].devname, "/dev/%s%d", GRAPE7E_DEVNAME, j);
                hib[i].type = HIB_GRAPE7E;
            }
	}
    }
}

static void
lock_hib(int devid)
{
    int locked; // 0:success  -1:failure
    int cnt = 0;

    locked = lockf(hib_fd[devid], F_TLOCK, 0);
    if (locked == 0) return; // returns successfully.

    if ((errno == EACCES) || (errno == EAGAIN)) {
	warn(1, "Someone is using hib[%d]. Sleep...\n", devid);
    }
    else {
	perror("hib_openMC");
	exit(2);
    }
    locked = lockf(hib_fd[devid], F_LOCK, 0);
}

static void
unlock_hib(int devid)
{
    if (lockf(hib_fd[devid], F_ULOCK, 0)) {
	perror("unlock_hibMC");
	exit (2);
    }
}
