#include <stdio.h>
#include <unistd.h>
#include "hibutil.h"

static char* productname[] = {
    "",
    "GRAPE-7",
    "GRAPE-7E",
    "GRAPE-DR",
};

static char* modelname[][16] = {
    {},
    { "", "Model100", "Model300D", "Model300", ""   , "", "Model600", "", "Model800"},
    { "", "Model100", "",          ""        , ""   , "", ""        , "", },
    { "", "TB1"     , "TB2"      , "TB3"     , "TB4",                 },
};

static char* backendname[] = {
    "empty",
    "G5   ",
    "G6   ",
};

void g5info(UINT32 binfo);
void noinfo(UINT32 binfo);

static void (*longinfo[])(UINT32 binfo) = {
    noinfo,
    g5info,
    noinfo,
};


static unsigned int modelid2nchip[] = {
    0,
    1, // m100
    3, // m300d
    3, // m300
    0,
    0,
    6, // m600
    0,
    1, // m800
};

void
g5info(UINT32 binfo)
{
    int npipe    =   (binfo >>  0) & 0xff;
    int jmem     = 2048 << ((binfo>>8) & 0x3);
    int p3m      =   (binfo >> 15) & 0x1;
    int nb       =   (binfo >> 16) & 0x1;
    int foutfifo = (((binfo >> 10) & 0x3) << 10);
    int eps2     =   (binfo >> 14) & 0x1;
    int modelid = (binfo >> 24) & 0xf;
    int nchip    = modelid2nchip[modelid];

    printf("        number of chips             : %d\n", nchip);
    printf("        number of pipelines/chip    : %d\n", npipe);
    printf("        j-particle memory size/chip : %d particles\n", jmem);
    printf("        P3M cutoff                  : %s\n", p3m ?  "available"      : "not available");
    printf("        neighbor search             : %s\n", nb  ?  "available"      : "not available");
    printf("        fout fifo size              : %d bytes\n", foutfifo * sizeof(UINT64));
    printf("        number format               : %s\n", eps2 ? "floating-point" : "logarithmic");
}

void
noinfo(UINT32 binfo)
{
    printf("        no verbose info available.\n");
}

static void
showusage(char *cmdname)
{
    fprintf(stderr, "'%s' shows GRAPE devices available.\n", cmdname);
    fprintf(stderr, "usage: %s [options]\n", cmdname);
    fprintf(stderr, "    -l            be verbose.\n");
    fprintf(stderr, "    -v            same as -l.\n");
    fprintf(stderr, "    -d num        print num-th device only\n"
	            "                  (print all devices by default).\n");
    fprintf(stderr, "    -h            print this message.\n");
}

int
main(int argc, char **argv)
{
    Hib *h;
    UINT32 binfo;
    int c, ic;
    int devid = -1;
    int productid, modelid, backendid;
    char* param = "d:lhv";
    static is_long_format = 0;

    while ((c = getopt(argc, argv, param)) != EOF) {
	switch (c) {
	  case 'l':
	  case 'v':
	    is_long_format = 1;
	    break;
	  case 'd':
	    devid = atoi(optarg);
	    if (devid < 0 || NHIB <= devid) {
		fprintf(stderr, "invalid device ID (%d)\n", devid);
		exit(1);
	    }
	    break;
	  case 'h':
	    showusage(argv[0]);
	    exit(0);
	    break;
	}	
    }

    printf("devid grape(model)              backend-logic\n");
    for (ic = 0; ic < NHIB; ic++) {

	if (devid != -1 && devid != ic) continue;

        h = hib_openMC(ic);
        binfo = hib_mem_readMC(ic, h->boardinfo);

        productid = (binfo >> 28) & 0xf;
        modelid = (binfo >> 24) & 0xf;
        backendid = (binfo >> 20) & 0xf;
        hib_closeMC(ic);

	char buf[256];
	sprintf(buf, "%s(%s)", productname[productid], modelname[productid][modelid]);
	printf("% 2d    %-25s %s\n",
	       ic, buf, backendname[backendid]);
	if (is_long_format) {
	    (*longinfo[backendid])(binfo);
	}
    }

#if 0
    int i, j;
    for (i = 0; i < 4; i++) {
	for (j = 0; j < 16; j++) {
	    printf("%s  ",modelname[i][j]);
	}
	printf("\n");
    }
#endif
}
