#ifndef HIBUTIL_H
#define HIBUTIL_H

#include <grape7x.h>
#include <grape7e.h>
#include "typedef.h"

#if defined(__cplusplus)
extern "C"
{
#endif

#define NHIB (NGRAPE7X+NGRAPE7E)

    enum {
        TESTMODE_NONE=0,         // normal operation.
	TESTMODE_REFDESIGN_FIFO, // data received is directly put into internal FIFO.
	TESTMODE_REFDESIGN_RAM,  // data received is ignored.
    };
    enum {
        SENDFUNC_DMAR = 0,
	SENDFUNC_PIOW,
    };
    enum {
        HIB_GRAPE7X = 1,
        HIB_GRAPE7E = 2,
    };
    enum {
        DESIGN_EMPTY = 0,
        DESIGN_G5 = 1,
        DESIGN_G6 = 2,
    };

    /* Hib descriptor */
    typedef struct {
        /*
         * card attributes
         */
        int type;             // 0x1:HIB_GRAPE7X 0x2:HIB_GRAPE7E
        int backend_design;   // 0x0:DESIGN_EMPTY 0x1:DESIGN_G5 0x2:DESIGN_G6
        char devname[256];    // device name (eg. /dev/grape7x)

        /*
         * address space mapped to the user space
         */
        // HIB local memory space:
        volatile UINT32 *local_reg;    // local register
        volatile UINT64 *piow_dblbuf;  // double buffer for PIO write transfer
        volatile UINT64 *backend;      // backend space

        // kernel space:
        UINT64 *dmar_buf;     // buffer for DMA read transfer (host -> backend mem).
        UINT64 *dmaw_buf;     // buffer for DMA write transfer (host <- backend mem).

        /*
         * HIB resource sizes
         */
        unsigned int dmabuf_bytes;
        unsigned int piowbuf_bytes;
        unsigned int local_mem_bytes;
        unsigned int backend_mem_bytes;

        /*
         * HIB ioctl commands
         */
        unsigned int write_cfg;
        unsigned int read_cfg;
        unsigned int get_dmaw_pa;
        unsigned int get_dmar_pa;
        unsigned int set_map_mode;

        unsigned int map_hibmem;
        unsigned int map_dmarbuf;
        unsigned int map_dmawbuf;
        unsigned int map_backend;
        unsigned int map_dblbuf;

        /*
         * HIB local register address
         */
        UINT32 boardinfo;
        UINT32 dma0lsb;
        UINT32 dma0msb;
        UINT32 dma0size;
        UINT32 dma0cmd;
        UINT32 dma0misc;
        UINT32 dma1lsb;
        UINT32 dma1msb;
        UINT32 dma1size;
        UINT32 dma1cmd;
        UINT32 dma1misc;
        UINT32 dmastat;
        UINT32 mailbox;
        UINT32 asmi;
        UINT32 pllconf;

        /*
         * HIB local register bit offset
         */
        unsigned int dma0cmd_kickoff;
        unsigned int dma0misc_done_bit;
        unsigned int dma0misc_swap_sram_bit;
        unsigned int dma0misc_sram_wcnt_bit;
        unsigned int dma1cmd_kickoff;
        unsigned int dma1misc_done_bit;
        unsigned int dmastat_dma_reset_bit;
        unsigned int dmastat_reset_backend_bit;
        unsigned int asmi_error_bit;
        unsigned int asmi_busy_bit;
	UINT32 asmi_cmd_rsid;
	UINT32 asmi_cmd_wp;
	UINT32 asmi_cmd_eb;
        unsigned int pllconf_write_bit;
        unsigned int pllconf_reconfig_bit;
        unsigned int pllconf_busy_bit;

    } Hib;

    /*
     * global function(s)
     */
    Hib *hib_openMC(int devid);
    void hib_closeMC(int devid);
    void hib_set_test_modeMC(int devid, int mode);
    void hib_set_warn_level(int lv);

    void hib_sendMC(int devid, int size, UINT64 *buf);
    void hib_set_sendfuncMC(int devid, int func); // SENDFUNC_DMAR or SENDFUNC_PIOW
    void hib_piowMC(int devid, int size, UINT64 *buf);
    void hib_dmarMC(int devid, int size, UINT64 *buf);

    int hib_recvMC(int devid, int size, UINT64 *buf);
    int hib_dmawMC(int devid, int size, UINT64 *buf);
    void hib_start_dmawMC(int devid, int size, UINT64 *buf);
    int hib_finish_dmawMC(int devid);

    UINT32 hib_config_readMC(int devid, UINT32 addr);
    void hib_config_writeMC(int devid, UINT32 addr, UINT32 value);
    UINT32 hib_mem_readMC(int devid, UINT32 addr);
    void hib_mem_writeMC(int devid, UINT32 addr, UINT32 value);

#if defined(__cplusplus)
}
#endif

#endif /* HIBUTIL_H */
