#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "g5util.h"

static void
set_range(int devid, double xmin, double xmax)
{
    double xsize = xmax - xmin;
    double xscale = pow(2.0, 32.0) / xsize;
    double vscale = 1.0;
    double eps2scale = xscale * xscale;
    double mscale = 1.0;
    double ascale = xscale * xscale / mscale;

    g5_set_range_xjMC(devid, xmin, xmax);
    g5_set_range_xiMC(devid, xmin, xmax);
    g5_set_scale_eps2MC(devid, eps2scale);
    g5_set_scale_mjMC(devid, mscale);
    g5_set_scale_aMC(devid, ascale);
}

void
force_grape(int devid, double (*x)[3], double (*v)[3], double *m,
	    double *eps2, int *fshift, int *jshift, double (*a)[3], double (*jerk)[3], int n)
{
    int off, ni, np, i, k;
    double ascale, jscale;

    g5_set_jpMC(devid, 0, n, x, m);
    g5_set_nMC(devid, n);

    ni = g5_get_number_of_pipelinesMC(devid);
    for (off = 0; off < n; off += ni) {
        if (off + ni > n) {
            ni = n - off;
        }
        g5_set_ipMC(devid, ni, (double (*)[3])x[off], eps2 + off);
        g5_runMC(devid);
        g5_get_foutMC(devid, ni, (double (*)[3])a[off]);
	for (i = 0; i < ni; i++) {

            ascale = pow(2.0, -fshift[off + i]);

	    for (k = 0; k < 3; k++) {
		a[off + i][k] *= ascale;
	    }
	    jscale = pow(2.0, -jshift[off + i]);
	    for (k = 0; k < 3; k++) {
		jerk[off + i][k] *= jscale;
	    }
	}
    }
}

#define NMAX (4096)

int
main(int argc, char **argv)
{
    double m[NMAX], x[NMAX][3], v[NMAX][3], eps2[NMAX];
    int    fshift[NMAX], jshift[NMAX];
    double a[NMAX][3], jerk[NMAX][3];
    int    devid, n, i;
    double xmax;

    if (argc < 2) {
	fprintf(stderr, "usage: %s <card ID>\n", argv[0]);
	exit(1);
    }
    devid = atoi(argv[1]);
    fprintf(stderr, "force_grape() uses g5[%d].\n", devid);

    n = 2;

    x[0][0] = +0.5;
    x[0][1] = +0.0;
    x[0][2] = +0.0;

    x[1][0] = -0.5;
    x[1][1] = -0.0;
    x[1][2] = -0.0;

    v[0][0] = +0.5;
    v[0][1] = +0.5;
    v[0][2] = +0.5;

    v[1][0] = -0.5;
    v[1][1] = -0.5;
    v[1][2] = -0.5;

#define FSHIFT (75)

    for (i = 0; i < n; i++) {
	eps2[i] = 1e-6;
	m[i] = 1.0;
	fshift[i] = FSHIFT;
    }
    xmax = 64.0;

    for (i = 0; i < n; i++) {
	printf("  m[%d]      : %+5.3f\n", i, m[i]);
	printf("  eps2[%d]   : %+5.3f\n", i, eps2[i]);
	printf("  x[%d]      : %+5.3f %+5.3f %+5.3f\n",
	       i, x[i][0], x[i][1], x[i][2]);
	printf("  v[%d]      : %+5.3f %+5.3f %+5.3f\n",
	       i, v[i][0], v[i][1], v[i][2]);
	printf("  fshift[%d] : %d\n", i, fshift[i]);
	printf("  jshift[%d] : %d\n", i, jshift[i]);
	printf("\n");
    }

    g5_openMC(devid);
    set_range(devid, -xmax, +xmax);
    force_grape(devid, x, v, m, eps2, fshift, jshift, a, jerk, n);
    g5_closeMC(devid);

    printf("\n");
    for (i = 0; i < n; i++) {
	printf("  a[%d]      : %+5.3f %+5.3f %+5.3f\n",
	       i, a[i][0], a[i][1], a[i][2]);
        printf("  jerk[%d]   : %+5.3f %+5.3f %+5.3f\n",
               i, jerk[i][0], jerk[i][1], jerk[i][2]);
	printf("\n");
    }
}
