#include <stdio.h>
#include <stdlib.h>
#include <sys/resource.h>
#include <sys/time.h>

void
create_cold_homosphere(int n, double *mj, double (*xj)[3], double (*vj)[3])
{
    int i;
    double mass = 1.0/n;

    for (i = 0; i < n; i++) {
	mj[i] = mass;
    }
    for (i = 0; i < n; i++) {
	vj[i][0] = 0.0;
	vj[i][1] = 0.0;
	vj[i][2] = 0.0;
    }
    i = 0;
    while (i < n) {
	double x, y, z;
	x = 2.0*drand48()-1.0;
	y = 2.0*drand48()-1.0;
	z = 2.0*drand48()-1.0;
	if (x*x+y*y+z*z<1.0) {
	    xj[i][0] = x;
	    xj[i][1] = y;
	    xj[i][2] = z;
	    i++;
	}
    }
}

void
readnbody(int *nj, double *mj, double (*xj)[3], double (*vj)[3], char *fname)
{
     int i, dummy;
     double dummyd;
     FILE *fp;

     fp = fopen(fname, "r");
     if (fp == NULL) {
	  perror("readnbody");
	  exit(1);
     }
     fscanf(fp, "%d\n", nj);
     fscanf(fp, "%d\n", &dummy);
     fscanf(fp, "%lf\n", &dummyd);
     fprintf(stderr, "nj: %d\n", *nj);
     for (i = 0; i < *nj; i++) {
	  fscanf(fp, "%lf\n", mj+i);
     }
     for (i = 0; i < *nj; i++) {
	  fscanf(fp, "%lf %lf %lf\n",
		 xj[i]+0, xj[i]+1, xj[i]+2);
     }
     for (i = 0; i < *nj; i++) {
	  fscanf(fp, "%lf %lf %lf\n",
		 vj[i]+0, vj[i]+1, vj[i]+2);
     }
}

void
writenbody(int nj, double *mj, double (*xj)[3], double (*vj)[3], char *fname)
{
     int i, dummy;
     FILE *fp;

     fp = fopen(fname, "w");
     fprintf(fp, "%d\n", nj);
     fprintf(fp, "%d\n", 3);
     fprintf(fp, "%e\n", 0.0);
     for (i = 0; i < nj; i++) {
	  fprintf(fp, " % 15.13E\n", mj[i]);
     }
     for (i = 0; i < nj; i++) {
         fprintf(fp, " % 15.13E % 15.13E % 15.13E\n",
		  xj[i][0], xj[i][1], xj[i][2]);
     }
     for (i = 0; i < nj; i++) {
         fprintf(fp, " % 15.13E % 15.13E % 15.13E\n",
                 vj[i][0], vj[i][1], vj[i][2]);
     }
}

void
push_velocity(double (*vj)[3], double (*a)[3], double dt, int nj)
{
     int j, k;

     for (j = 0; j < nj; j++) {
	  for (k = 0; k < 3; k++) {
	       vj[j][k] += dt * a[j][k];
	  }
     }
}

void
push_position(double (*xj)[3], double (*vj)[3], double (*a)[3],
	      double dt, int nj)
{
     int j, k;

     for (j = 0; j < nj; j++) {
	  for (k = 0; k < 3; k++) {
	       xj[j][k] += dt * vj[j][k];
	  }
     }
}

void
energy(double *mj, double (*vj)[3], double *p, int nj, double *ke, double *pe)
{
     int i, k;
     
     *pe = 0;
     *ke = 0;
     for (i = 0; i < nj; i++) {
	  *pe += mj[i] * p[i];
	  for (k = 0; k < 3; k++) {
	       *ke += 0.5 * mj[i] * vj[i][k] * vj[i][k];
	  }
     }
     *pe /= 2.0;
}

#if 0

/* CPU time */
void
get_cputime(double *lap, double *split)
{
    struct rusage x;
    double sec,microsec;

    getrusage(RUSAGE_SELF,&x);
    sec = x.ru_utime.tv_sec + x.ru_stime.tv_sec ;
    microsec = x.ru_utime.tv_usec + x.ru_stime.tv_usec ;

    *lap = sec + microsec / 1000000.0 - *split;
    *split = sec + microsec / 1000000.0;
}

#else

/* elapsed time in real world */
void
get_cputime(double *lap, double *split)
{
    struct timeval x;
    double sec,microsec;

    gettimeofday(&x, NULL);
    sec = x.tv_sec;
    microsec = x.tv_usec;

    *lap = sec + microsec / 1000000.0 - *split;
    *split = sec + microsec / 1000000.0;
}

#endif
