#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "g6util.h"

#define JSHIFT (190)
#define VSCALE pow(2.0, 10.0);

static void
set_range(int devid, double xmin, double xmax)
{
    double xsize = xmax - xmin;
    double xscale = pow(2.0, 64.0) / xsize;
    double vscale = VSCALE;
    double eps2scale = xscale * xscale;
    double mscale = 1.0;
    double ascale = xscale * xscale / mscale;
    double jscale = xscale * xscale * xscale / mscale / vscale;

    g6_set_range_xjMC(devid, xmin, xmax);
    g6_set_range_xiMC(devid, xmin, xmax);
    g6_set_scale_vjMC(devid, vscale);
    g6_set_scale_viMC(devid, vscale);
    g6_set_scale_epsi2MC(devid, eps2scale);
    g6_set_scale_mjMC(devid, mscale);
    g6_set_scale_accMC(devid, ascale);
    g6_set_scale_jerkMC(devid, jscale);
}

void
force_grape(int devid, double (*x)[3], double (*v)[3], double *m,
	    double *eps2, int *fshift, int *jshift, double (*a)[3], double (*jerk)[3], int n)
{
    int off, ni, np, i, k;
    double ascale, jscale;

    g6_set_jpMC(devid, 0, n, m, x, v);
    g6_set_nMC(devid, n);

    ni = g6_get_number_of_pipelinesMC(devid);
    for (off = 0; off < n; off += ni) {
        if (off + ni > n) {
            ni = n - off;
        }
        g6_set_ipMC(devid, ni, (double (*)[3])x[off], (double (*)[3])v[off], eps2 + off,
                    fshift + off, jshift + off);
        g6_runMC(devid);
        g6_get_foutMC(devid, ni, (double (*)[3])a[off], (double (*)[3])jerk[off]);
	for (i = 0; i < ni; i++) {
	    ascale = pow(2.0, -fshift[off + i]);
	    for (k = 0; k < 3; k++) {
		a[off + i][k] *= ascale;
	    }
	    jscale = pow(2.0, -jshift[off + i]);
	    for (k = 0; k < 3; k++) {
		jerk[off + i][k] *= jscale;
	    }
	}
    }
}

#define NMAX (4096)

int
main(int argc, char **argv)
{
    double m[NMAX], x[NMAX][3], v[NMAX][3], eps2[NMAX];
    int    fshift[NMAX], jshift[NMAX];
    double a[NMAX][3], jerk[NMAX][3];
    int    devid, n, i;
    double xmax;

    if (argc < 2) {
	fprintf(stderr, "usage: %s <card ID>\n", argv[0]);
	exit(1);
    }
    devid = atoi(argv[1]);
    fprintf(stderr, "force_grape() uses g6[%d].\n", devid);

    n = 2;

    x[0][0] = +0.5;
    x[0][1] = +0.517;
    x[0][2] = +0.29036;

    x[1][0] = -0.34232;
    x[1][1] = -0.235554;
    x[1][2] = -0.89655;


    v[0][0] = +0.125;
    v[0][1] = +0.8349;
    v[0][2] = +0.298742984;

    v[1][0] = -0.5218746528;
    v[1][1] = -0.743722842;
    v[1][2] = -0.62432;

    for (i = 0; i < n; i++) {
	eps2[i] = 0.0;
	m[i] = 1.0;
	fshift[i] = 150;
	jshift[i] = JSHIFT;
    }

    xmax = 64.0;

    for (i = 0; i < n; i++) {
	printf("  m[%d]      : %+5.3f\n", i, m[i]);
	printf("  eps2[%d]   : %+5.3f\n", i, eps2[i]);
	printf("  x[%d]      : %+5.3f %+5.3f %+5.3f\n",
	       i, x[i][0], x[i][1], x[i][2]);
	printf("  v[%d]      : %+5.3f %+5.3f %+5.3f\n",
	       i, v[i][0], v[i][1], v[i][2]);
	printf("  fshift[%d] : %d\n", i, fshift[i]);
	printf("  jshift[%d] : %d\n", i, jshift[i]);
	printf("\n");
    }

    g6_openMC(devid);
    set_range(devid, -xmax, +xmax);
    force_grape(devid, x, v, m, eps2, fshift, jshift, a, jerk, n);
    g6_closeMC(devid);

    printf("\n");
    for (i = 0; i < n; i++) {
	printf("  a[%d]      : %+5.3f %+5.3f %+5.3f\n",
	       i, a[i][0], a[i][1], a[i][2]);
        printf("  jerk[%d]   : %+5.3f %+5.3f %+5.3f\n",
               i, jerk[i][0], jerk[i][1], jerk[i][2]);
	printf("\n");
    }
}
