--
-- cg: A clock generation module to provide programmable clock signals
-- to backend pipelines. It also provides a clock signal used for an
-- interface logic to the host.
--

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
library lpm;
use lpm.lpm_components.all;

entity cg is
  port (
    -- a reference clock externally provided.
    ext_clk            : in std_logic;
    
    -- clock signals this module provides.
    clk0               : out std_logic;  -- interface clk (non programmable)
    clk1               : out std_logic;  -- backend clk (programmable)

    -- PLL reconfiguration control signals.
    reset              : in  std_logic;  -- reset this module. active high.
    wrreq              : in  std_logic;  -- pll configuration parameter write enable.
    datain             : in  std_logic_vector(24 downto 0); -- pll configuration parameter.
    busy               : out std_logic   -- pll configuration is in progress.
  );
end cg;

architecture structural of cg is

  component cgpllconf
    port ( 
      busy             : out  std_logic;
      clock            : in  std_logic;
      counter_param    : in  std_logic_vector (2 downto 0) := (others => '0');
      counter_type     : in  std_logic_vector (3 downto 0) := (others => '0');
      data_in          : in  std_logic_vector (8 downto 0) := (others => '0');
      data_out         : out  std_logic_vector (8 downto 0);
      pll_areset       : out  std_logic;
      pll_areset_in    : in  std_logic := '0';
      pll_configupdate : out  std_logic;
      pll_scanclk      : out  std_logic;
      pll_scanclkena   : out  std_logic;
      pll_scandata     : out  std_logic;
      pll_scandataout  : in  std_logic := '0';
      pll_scandone     : in  std_logic := '0';
      read_param       : in  std_logic := '0';
      reconfig         : in  std_logic := '0';
      reset            : in  std_logic;
      write_param      : in  std_logic := '0'
    ); 
  end component;

  component cgdcfifo
    port (
      aclr     : in std_logic := '0';
      data     : in std_logic_vector (24 downto 0);
      rdclk    : in std_logic;
      rdreq    : in std_logic;
      wrclk    : in std_logic;
      wrreq    : in std_logic;
      q        : out std_logic_vector (24 downto 0);
      rdempty  : out std_logic 
    );
  end component;

  component cgpll0
    port (
        inclk0         : in std_logic  := '0';
        c0             : out std_logic;
        c1             : out std_logic
    );
  end component;

  component cgpll1
    port (
        areset         : in std_logic  := '0';
        configupdate   : in std_logic  := '0';
        inclk0         : in std_logic  := '0';
        scanclk        : in std_logic  := '1';
        scanclkena     : in std_logic  := '0';
        scandata       : in std_logic  := '0';
        c0             : out std_logic ;
        scandataout    : out std_logic ;
        scandone       : out std_logic 
    );
  end component;

  signal pcclk, lclk0, lclk1  : std_logic;
  signal pf_rdempty           : std_logic;
  signal pf_rdreq, pf_rdreq1  : std_logic;
  signal pf_data              : std_logic_vector(24 downto 0);
  signal pc_areset            : std_logic;
  signal pc_configupdate      : std_logic;
  signal pc_scanclk           : std_logic;
  signal pc_scanclkena        : std_logic;
  signal pc_scandata          : std_logic;
  signal pc_scandataout       : std_logic;
  signal pc_scandone          : std_logic;
  signal write_param          : std_logic;
  signal reconfig             : std_logic;
  
begin
  
  clk0 <= lclk0;
  clk1 <= lclk1;

  cgpll0_instance : cgpll0
    port map (
      inclk0  => ext_clk,
      c0      => lclk0,        -- interface clk
      c1      => pcclk         -- slower clk (ext_clk*2/3 MHz) internally used to configure cgpll1.
    );

  cgpll1_instance : cgpll1
    port map (
      inclk0       => lclk0,             -- interface clk.
      c0           => lclk1,             -- a programmable backend clk.
      areset       => pc_areset,
      configupdate => pc_configupdate,
      scanclk      => pc_scanclk,
      scanclkena   => pc_scanclkena,
      scandata     => pc_scandata,
      scandataout  => pc_scandataout,
      scandone     => pc_scandone
    );

  --
  -- dual clock fifo
  -- write in from PCI-X, read out to pllconf.
  --
  cgdcfifo_instance : cgdcfifo
    port map (
      aclr      => reset,
      rdempty   => pf_rdempty,

      wrclk     => lclk0,
      wrreq     => wrreq,
      data      => datain,

      rdclk     => pcclk,
      rdreq     => pf_rdreq,
      q         => pf_data
      );

  pf_rdreq <= not pf_rdempty;

  process (pcclk)
  begin
    if rising_edge (pcclk) then
      pf_rdreq1 <= pf_rdreq;
      if (pf_rdreq1 = '1') then
        write_param <= pf_data(0);
        reconfig    <= pf_data(2);
      else
        write_param <= '0';
        reconfig    <= '0';
      end if;
    end if;
  end process; 

  cgpllconf_instance : cgpllconf
    port map (
      clock            => pcclk,
      busy             => busy,
      data_in          => pf_data(24 downto 16),
      counter_type     => pf_data(11 downto  8),
      counter_param    => pf_data( 5 downto  3),
      reconfig         => reconfig,
      read_param       => '0',
      write_param      => write_param,
      pll_areset_in    => reset,
      pll_areset       => pc_areset,
      pll_configupdate => pc_configupdate,
      pll_scanclk      => pc_scanclk,
      pll_scanclkena   => pc_scanclkena,
      pll_scandata     => pc_scandata,
      pll_scandataout  => pc_scandataout,
      pll_scandone     => pc_scandone,
      reset            => reset
    );

end structural;
