library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
library lpm;
use lpm.lpm_components.all;

entity hib is
  generic (
    DATA_PATH_SIZE      : integer := 64; -- 64-bit word
    SRAM_ADDR_WIDTH     : integer := 9;  -- 512 words
    FIFO_SIZE           : integer := 9  -- 512 words
    );
  port (
    -- PCIe signals
    osc_local_32         : in std_logic;  -- GXII
    clk100_ext           : in std_logic;
    mperst               : in std_logic;
    rx_in0               : in std_logic;
    tx_out0              : out std_logic;
    rx_in1               : in std_logic;
    tx_out1              : out std_logic;
    rx_in2               : in std_logic;
    tx_out2              : out std_logic;
    rx_in3               : in std_logic;
    tx_out3              : out std_logic;
    wake                 : out std_logic;

    -- PLDA board specific signals
    prot0_out            : out std_logic;    
    prot2_in             : in std_logic_vector(1 downto 0);
    prot2_out            : out std_logic;    
    usr_sw               : in  std_logic_vector (3 downto 0);
    usr_led              : out std_logic_vector (3 downto 0);

    -- global lines
    clk_out              : out std_logic;

    -- interface to the backend 
    hib_we               : out std_logic;
    hib_data             : out std_logic_vector(63 downto 0);
    backend_we           : in std_logic;
    backend_data         : in std_logic_vector(63 downto 0);
    reset_backend        : out std_logic;

    -- board information register initial value
    board_info           : in std_logic_vector(31 downto 0)
    );
end hib;

architecture structural of hib is

  component siigx125_phyx4                        -- GXII
    port (
      osc_local_32            : in    std_logic;  -- GXII
      cal_blk_clk             : in    std_logic;  -- GXII
      clk125plllock           : out   std_logic;  -- GXII
      clk100                    : in    std_logic;
      clk125out                 : out     std_logic;
--      clk100out                 : out    std_logic;  -- DKit
      rstn                      : in    std_logic;
      npor                      : in    std_logic;

      rx_in0                    : in    std_logic;
      tx_out0                   : out    std_logic;
      rx_in1                    : in    std_logic;
      tx_out1                   : out    std_logic;
      rx_in2                    : in    std_logic;
      tx_out2                   : out    std_logic;
      rx_in3                    : in    std_logic;
      tx_out3                   : out    std_logic;

      -- PIPE interface
      wake                      : out    std_logic;
      phystatus                 : out    std_logic;
      powerdown                 : in    std_logic_vector(1 downto 0);
      txdetectrx                : in    std_logic;

      txdata0                   : in    std_logic_vector(15 downto 0);
      txdatak0                  : in    std_logic_vector(1 downto 0);
      txelecidle0               : in    std_logic;
      txcompl0                  : in    std_logic;
      rxpolarity0               : in    std_logic;
      rxdata0                   : out    std_logic_vector(15 downto 0);
      rxdatak0                  : out    std_logic_vector(1 downto 0);
      rxvalid0                  : out    std_logic;
      rxelecidle0               : out    std_logic;
      rxstatus0                 : out    std_logic_vector(2 downto 0);

      txdata1                   : in    std_logic_vector(15 downto 0);
      txdatak1                  : in    std_logic_vector(1 downto 0);
      txelecidle1               : in    std_logic;
      txcompl1                  : in    std_logic;
      rxpolarity1               : in    std_logic;
      rxdata1                   : out    std_logic_vector(15 downto 0);
      rxdatak1                  : out    std_logic_vector(1 downto 0);
      rxvalid1                  : out    std_logic;
      rxelecidle1               : out    std_logic;
      rxstatus1                 : out    std_logic_vector(2 downto 0);

      txdata2                   : in    std_logic_vector(15 downto 0);
      txdatak2                  : in    std_logic_vector(1 downto 0);
      txelecidle2               : in    std_logic;
      txcompl2                  : in    std_logic;
      rxpolarity2               : in    std_logic;
      rxdata2                   : out    std_logic_vector(15 downto 0);
      rxdatak2                  : out    std_logic_vector(1 downto 0);
      rxvalid2                  : out    std_logic;
      rxelecidle2               : out    std_logic;
      rxstatus2                 : out    std_logic_vector(2 downto 0);

      txdata3                   : in    std_logic_vector(15 downto 0);
      txdatak3                  : in    std_logic_vector(1 downto 0);
      txelecidle3               : in    std_logic;
      txcompl3                  : in    std_logic;
      rxpolarity3               : in    std_logic;
      rxdata3                   : out    std_logic_vector(15 downto 0);
      rxdatak3                  : out    std_logic_vector(1 downto 0);
      rxvalid3                  : out    std_logic;
      rxelecidle3               : out    std_logic;
      rxstatus3                 : out    std_logic_vector(2 downto 0)
      );
  end component;

  component pciecorex4
    port(
      clk : in std_logic;
      rstn : in std_logic;
      srst : in std_logic;
      npor : in std_logic;
      test_mode : in std_logic_vector(15 downto 0);
      rstn_out : out std_logic;
      npor_out : out std_logic;
      phystatus : in std_logic;
      powerdown : out std_logic_vector(1 downto 0);
      txdetectrx : out std_logic;
      txdata0 : out std_logic_vector(15 downto 0);
      txdatak0 : out std_logic_vector(1 downto 0);
      txelecidle0 : out std_logic;
      txcompl0 : out std_logic;
      rxpolarity0 : out std_logic;
      rxdata0 : in std_logic_vector(15 downto 0);
      rxdatak0 : in std_logic_vector(1 downto 0);
      rxvalid0 : in std_logic;
      rxelecidle0 : in std_logic;
      rxstatus0 : in std_logic_vector(2 downto 0);
      txdata1 : out std_logic_vector(15 downto 0);
      txdatak1 : out std_logic_vector(1 downto 0);
      txelecidle1 : out std_logic;
      txcompl1 : out std_logic;
      rxpolarity1 : out std_logic;
      rxdata1 : in std_logic_vector(15 downto 0);
      rxdatak1 : in std_logic_vector(1 downto 0);
      rxvalid1 : in std_logic;
      rxelecidle1 : in std_logic;
      rxstatus1 : in std_logic_vector(2 downto 0);
      txdata2 : out std_logic_vector(15 downto 0);
      txdatak2 : out std_logic_vector(1 downto 0);
      txelecidle2 : out std_logic;
      txcompl2 : out std_logic;
      rxpolarity2 : out std_logic;
      rxdata2 : in std_logic_vector(15 downto 0);
      rxdatak2 : in std_logic_vector(1 downto 0);
      rxvalid2 : in std_logic;
      rxelecidle2 : in std_logic;
      rxstatus2 : in std_logic_vector(2 downto 0);
      txdata3 : out std_logic_vector(15 downto 0);
      txdatak3 : out std_logic_vector(1 downto 0);
      txelecidle3 : out std_logic;
      txcompl3 : out std_logic;
      rxpolarity3 : out std_logic;
      rxdata3 : in std_logic_vector(15 downto 0);
      rxdatak3 : in std_logic_vector(1 downto 0);
      rxvalid3 : in std_logic;
      rxelecidle3 : in std_logic;
      rxstatus3 : in std_logic_vector(2 downto 0);
      prot0_out : out std_logic;
      prot2_in : in std_logic_vector(1 downto 0);
      prot2_out : out std_logic;
      cfg_prmcsr : out std_logic_vector(31 downto 0);
      cfg_devcsr : out std_logic_vector(31 downto 0);
      cfg_linkcsr : out std_logic_vector(31 downto 0);
      cfg_ltssm : out std_logic_vector(4 downto 0);
      slv_dataout : out std_logic_vector(63 downto 0);
      slv_bytevalid : out std_logic_vector(7 downto 0);
      slv_bytecount : out std_logic_vector(12 downto 0);
      slv_dwcount : out std_logic_vector(10 downto 0);
      slv_addr : out std_logic_vector(63 downto 0);
      slv_bar : out std_logic_vector(6 downto 0);
      slv_readreq : out std_logic;
      slv_cpladdr : out std_logic_vector(31 downto 0);
      slv_cplparam : out std_logic_vector(4 downto 0);
      slv_writereq : out std_logic;
      slv_write : out std_logic;
      slv_lastwrite : out std_logic;
      slv_accept : in std_logic;
      slv_abort : in std_logic;
      dma_rd : out std_logic;
      dma_rdaddr : out std_logic_vector(12 downto 0);
      dma_rdchannel : out std_logic_vector(7 downto 0);
      dma_rddata : in std_logic_vector(63 downto 0);
      dma_wr : out std_logic;
      dma_wraddr : out std_logic_vector(12 downto 0);
      dma_wrchannel : out std_logic_vector(7 downto 0);
      dma_wrdata : out std_logic_vector(63 downto 0);
      dma_wrbytevalid : out std_logic_vector(7 downto 0);
      dma0_regin : in std_logic_vector(127 downto 0);
      dma0_regout : out std_logic_vector(127 downto 0);
      dma0_param : in std_logic_vector(15 downto 0);
      dma0_control : in std_logic_vector(5 downto 0);
      dma0_status : out std_logic_vector(3 downto 0);
      dma0_fifocnt : in std_logic_vector(12 downto 0);
      dma1_regin : in std_logic_vector(127 downto 0);
      dma1_regout : out std_logic_vector(127 downto 0);
      dma1_param : in std_logic_vector(15 downto 0);
      dma1_control : in std_logic_vector(5 downto 0);
      dma1_status : out std_logic_vector(3 downto 0);
      dma1_fifocnt : in std_logic_vector(12 downto 0);
      dma2_regin : in std_logic_vector(127 downto 0);
      dma2_regout : out std_logic_vector(127 downto 0);
      dma2_param : in std_logic_vector(15 downto 0);
      dma2_control : in std_logic_vector(5 downto 0);
      dma2_status : out std_logic_vector(3 downto 0);
      dma2_fifocnt : in std_logic_vector(12 downto 0);
      dma3_regin : in std_logic_vector(127 downto 0);
      dma3_regout : out std_logic_vector(127 downto 0);
      dma3_param : in std_logic_vector(15 downto 0);
      dma3_control : in std_logic_vector(5 downto 0);
      dma3_status : out std_logic_vector(3 downto 0);
      dma3_fifocnt : in std_logic_vector(12 downto 0)
      );
  end component;

  component hibscfifo_sgx2
    generic
    (
      ADDR_WIDTH : INTEGER;
      DATA_WIDTH : INTEGER
    );
    port
    (
      show_ahead : in std_logic;
      clk : in std_logic;
      rstn : in std_logic;
      sclr : in std_logic;
      wrreq : in std_logic;
      wrdata : in std_logic_vector((DATA_WIDTH-1) downto 0);
      wrfull : out std_logic;
      wrempty : out std_logic;
      wrusedw : out std_logic_vector((ADDR_WIDTH-1) downto 0);
      rdreq : in std_logic;
      rddata : out std_logic_vector((DATA_WIDTH-1) downto 0);
      rdfull : out std_logic;
      rdempty : out std_logic;
      rdusedw : out std_logic_vector((ADDR_WIDTH-1) downto 0)
    );
  end component;

  component hibdcrambe_sgx2
    generic
    (     
        ADDR_WIDTH : integer;
        DATA_WIDTH : integer;
        BYTE_SIZE  : integer range 8 to 9:=8
    );
    port
    (
        wrclk     : in std_logic;
        wraddr    : in std_logic_vector (ADDR_WIDTH-1 downto 0);
        wrdata    : in std_logic_vector (DATA_WIDTH-1 downto 0);
        wrbe      : in std_logic_vector (DATA_WIDTH/BYTE_SIZE-1 downto 0);
        rdclk     : in std_logic;    
        rdaddr    : in std_logic_vector (ADDR_WIDTH-1 downto 0);
        rddata    : out std_logic_vector (DATA_WIDTH-1 downto 0)
    );
  end component;

  -- global lines
  signal clk125plllock,arst               : std_logic;  -- GXII
  signal clk,rstn                         : std_logic;
  signal rstn_cnt                         : std_logic_vector(19 downto 0);
  signal npor_out,rstn_out                : std_logic;
  signal sim_mode                         : std_logic;
  signal hardware_test_mode               : std_logic_vector (15 downto 0);

  -- PCIe signals
  signal phystatus                        : std_logic;
  signal powerdown                        : std_logic_vector(1 downto 0);
  signal txdetectrx                        : std_logic;

  signal txelecidle,rxelecidle            : std_logic_vector(3 downto 0);
  signal rxpolarity,txcompl,rxvalid          : std_logic_vector(3 downto 0);

  signal txdata0,rxdata0                    : std_logic_vector(15 downto 0);
  signal txdatak0,rxdatak0                : std_logic_vector(1 downto 0);
  signal rxstatus0                         : std_logic_vector(2 downto 0);

  signal txdata1,rxdata1                    : std_logic_vector(15 downto 0);
  signal txdatak1,rxdatak1                : std_logic_vector(1 downto 0);
  signal rxstatus1                         : std_logic_vector(2 downto 0);

  signal txdata2,rxdata2                    : std_logic_vector(15 downto 0);
  signal txdatak2,rxdatak2                : std_logic_vector(1 downto 0);
  signal rxstatus2                         : std_logic_vector(2 downto 0);

  signal txdata3,rxdata3                    : std_logic_vector(15 downto 0);
  signal txdatak3,rxdatak3                : std_logic_vector(1 downto 0);
  signal rxstatus3                         : std_logic_vector(2 downto 0);

  -- Slave interface
  signal slv_dataout: std_logic_vector(63 downto 0);
  signal slv_bytevalid: std_logic_vector(7 downto 0);
  signal slv_bytecount: std_logic_vector(12 downto 0);
  signal slv_dwcount: std_logic_vector(10 downto 0);
  signal slv_addr: std_logic_vector(63 downto 0);
  signal slv_bar: std_logic_vector(6 downto 0);
  signal slv_cpladdr: std_logic_vector(31 downto 0);
  signal slv_cplparam: std_logic_vector(4 downto 0);
  signal slv_writereq,slv_readreq: std_logic;
  signal slv_write,slv_lastwrite: std_logic;
  signal slv_accept,slv_abort: std_logic;
--  signal int_request: std_logic;

  -- DMA interface
  signal dma_wr,dma_rd: std_logic;
  signal dma_wraddr,dma_rdaddr: std_logic_vector(12 downto 0);
  signal dma_wrchannel,dma_rdchannel: std_logic_vector(7 downto 0);
  signal dma_wrdata,dma_rddata: std_logic_vector(63 downto 0);
  signal dma_wrbytevalid: std_logic_vector(7 downto 0);

  signal dma0_regin,dma1_regin,dma2_regin,dma3_regin: std_logic_vector(127 downto 0);
  signal dma0_regout,dma1_regout,dma2_regout,dma3_regout: std_logic_vector(127 downto 0);
  signal dma0_param,dma1_param,dma2_param,dma3_param: std_logic_vector(15 downto 0);
  signal dma0_control,dma1_control,dma2_control,dma3_control: std_logic_vector(5 downto 0);
  signal dma0_status,dma1_status,dma2_status,dma3_status: std_logic_vector(3 downto 0);
  signal dma0_fifocnt,dma1_fifocnt,dma2_fifocnt,dma3_fifocnt: std_logic_vector(12 downto 0);

  signal cfg_linkcsr: std_logic_vector(31 downto 0);

  -- for internal use
  signal res_mgt_slv_accept       : std_logic;
  signal sram_slv_accept          : std_logic;
  signal cpl_readreq2             : std_logic;
  signal cpl_readreq3             : std_logic;

  signal res_dma_rddata           : std_logic_vector(63 downto 0);
  signal fifo_wr_dma_wrdata       : std_logic_vector(63 downto 0);
  signal fifo_rd_dma_rddata       : std_logic_vector(63 downto 0);

  signal reset_backend_reg        : std_logic;

  signal dma_test_mode            : std_logic_vector(1 downto 0); -- EHIB test mode
  signal dma0_done,dma1_done      : std_logic;    
  signal swap_sram                : std_logic;    
  signal swap_done                : std_logic;    
  signal sram_wcnt                : std_logic_vector(SRAM_ADDR_WIDTH-1 downto 0);
  signal sram0_wlock,sram1_wlock  : std_logic;    
  signal sram0_done,sram1_done    : std_logic;    
  signal swap_sram_reg            : std_logic;
  signal board_info_reg           : std_logic_vector(31 downto 0);
  signal res_main_status_reg      : std_logic_vector(31 downto 0);
  signal res_mailbox_reg          : std_logic_vector(31 downto 0);
--  signal res_int_reg              : std_logic;
  signal dma_reset                : std_logic;
--  signal dma_int_req              : std_logic;
  signal sram_oe                        : std_logic;
  signal sram_rdata                     : std_logic_vector(63 downto 0);

  signal sram0_wlock_reg                : std_logic;
  signal sram0_wlockD                   : std_logic;
  signal sram0_we                       : std_logic_vector(7 downto 0);
  signal sram0_waddr                    : std_logic_vector(SRAM_ADDR_WIDTH-1 downto 0);
  signal sram0_wdata                    : std_logic_vector(63 downto 0);
  signal sram0_oe                       : std_logic;
  signal sram0_raddr                    : std_logic_vector(SRAM_ADDR_WIDTH-1 downto 0);
  signal sram0_rdata                    : std_logic_vector(63 downto 0);
  signal sram0_rcnt_dn                  : std_logic_vector(SRAM_ADDR_WIDTH-1 downto 0);
  signal sram0_rcnt_up                  : std_logic_vector(SRAM_ADDR_WIDTH-1 downto 0);

  signal sram1_wlock_reg                : std_logic;
  signal sram1_wlockD                   : std_logic;
  signal sram1_we                       : std_logic_vector(7 downto 0);
  signal sram1_waddr                    : std_logic_vector(SRAM_ADDR_WIDTH-1 downto 0);
  signal sram1_wdata                    : std_logic_vector(63 downto 0);
  signal sram1_oe                       : std_logic;
  signal sram1_raddr                    : std_logic_vector(SRAM_ADDR_WIDTH-1 downto 0);
  signal sram1_rdata                    : std_logic_vector(63 downto 0);
  signal sram1_rcnt_dn                  : std_logic_vector(SRAM_ADDR_WIDTH-1 downto 0);
  signal sram1_rcnt_up                  : std_logic_vector(SRAM_ADDR_WIDTH-1 downto 0);

  signal wr_fiforeq,rd_fiforeq          : std_logic;
  signal rd_fifosize                    : std_logic_vector(FIFO_SIZE-1 downto 0);
  signal dma1_status3_r                 : std_logic;
  signal dma0_status3_r                 : std_logic;
  signal dma0_control4,dma1_control4    : std_logic;         
  signal dma0_end,dma1_end              : std_logic;
  signal wr_fifodata                    : std_logic_vector(63 downto 0);
  
begin

  -- Set core test mode
  -- bit 0 => 0 for hardware implementation, 1 for simulation
  process (clk)
  begin
    sim_mode <='0';
    -- synthesis translate_off
    --    sim_mode <='1';
    -- synthesis translate_on
  end process;

  -- bits 7..5 => hardware test purpose
  hardware_test_mode <="00000000" & usr_sw(2 downto 0) & "0000" & sim_mode;

  -- Reset and POR management
  arst <= clk125plllock and mperst;     -- GXII
  process (clk,arst)                    -- GXII
  begin
    if arst='0' then                    -- GXII
      rstn_cnt <=(others=>'0');
      rstn <= '0';
    elsif rising_edge (clk) then
      -- Create a delay at power-up
      if rstn_out='0' or npor_out='0' then
        rstn <= '0';
      -- synthesis translate_off
      elsif rstn_cnt(9)='1' then 	-- 4us in simulation
        rstn <= '1';
      -- synthesis translate_on
      elsif rstn_cnt(19)='1' then	-- 4ms in real hardware
        rstn <= '1';
      else
        rstn_cnt <= unsigned(rstn_cnt) + '1';
      end if;
    end if;
  end process;

  process (clk,rstn)
  begin
    if rstn='0' then
      reset_backend <= '0';
    elsif rising_edge (clk) then
      if reset_backend_reg = '1' then
        reset_backend <= '0';
      else
        reset_backend <= '1';
      end if;
    end if;
  end process;

  -- PHY interface for StratixGX device
  clk_out <= clk;
  phy : siigx125_phyx4                    -- GXII
    port map (
      osc_local_32      => osc_local_32,  -- GXII free running clock for reset controller
      cal_blk_clk       => clk,           -- GXII should be common to all cal_blk_clk in the design!
      clk125plllock     => clk125plllock, -- GCII clk valid only when clk125plllock is high
      clk100            => clk100_ext,
      clk125out         => clk,
--      clk100out         => open,        -- DKit
      rstn              => rstn,
      npor              => mperst,

      rx_in0            => rx_in0,
      rx_in1            => rx_in1,
      rx_in2            => rx_in2,
      rx_in3            => rx_in3,
      tx_out0           => tx_out0,
      tx_out1           => tx_out1,
      tx_out2           => tx_out2,
      tx_out3           => tx_out3,
      wake                 => wake,

      phystatus         => phystatus,
      powerdown         => powerdown,
      txdetectrx        => txdetectrx,

      txdata0           => txdata0,
      txdatak0          => txdatak0,
      txelecidle0       => txelecidle(0),
      txcompl0          => txcompl(0),
      rxpolarity0       => rxpolarity(0),
      rxdata0           => rxdata0,
      rxdatak0          => rxdatak0,
      rxvalid0          => rxvalid(0),
      rxelecidle0       => rxelecidle(0),
      rxstatus0         => rxstatus0,

      txdata1           => txdata1,
      txdatak1          => txdatak1,
      txelecidle1       => txelecidle(1),
      txcompl1          => txcompl(1),
      rxpolarity1       => rxpolarity(1),
      rxdata1           => rxdata1,
      rxdatak1          => rxdatak1,
      rxvalid1          => rxvalid(1),
      rxelecidle1       => rxelecidle(1),
      rxstatus1         => rxstatus1,

      txdata2           => txdata2,
      txdatak2          => txdatak2,
      txelecidle2       => txelecidle(2),
      txcompl2          => txcompl(2),
      rxpolarity2       => rxpolarity(2),
      rxdata2           => rxdata2,
      rxdatak2          => rxdatak2,
      rxvalid2          => rxvalid(2),
      rxelecidle2       => rxelecidle(2),
      rxstatus2         => rxstatus2,

      txdata3           => txdata3,
      txdatak3          => txdatak3,
      txelecidle3       => txelecidle(3),
      txcompl3          => txcompl(3),
      rxpolarity3       => rxpolarity(3),
      rxdata3           => rxdata3,
      rxdatak3          => rxdatak3,
      rxvalid3          => rxvalid(3),
      rxelecidle3       => rxelecidle(3),
      rxstatus3         => rxstatus3
      );

  pciecorex4_inst : pciecorex4
    port map (
      clk => clk,
      rstn => rstn,
      srst => '0',                      -- soft reset not used for now
      npor => mperst,
      test_mode => hardware_test_mode,
      rstn_out => rstn_out,
      npor_out => npor_out,
      prot0_out => prot0_out,
      prot2_in => prot2_in,
      prot2_out => prot2_out, 

      phystatus => phystatus,
      powerdown => powerdown,
      txdetectrx => txdetectrx,

      txdata0	=> txdata0,
      txdatak0 => txdatak0,
      txelecidle0 => txelecidle(0),
      txcompl0 => txcompl(0),
      rxpolarity0 => rxpolarity(0),
      rxdata0 => rxdata0,
      rxdatak0 => rxdatak0,
      rxvalid0 => rxvalid(0),
      rxelecidle0 => rxelecidle(0),
      rxstatus0 => rxstatus0,

      txdata1	=> txdata1,
      txdatak1 => txdatak1,
      txelecidle1 => txelecidle(1),
      txcompl1 => txcompl(1),
      rxpolarity1 => rxpolarity(1),
      rxdata1 => rxdata1,
      rxdatak1 => rxdatak1,
      rxvalid1 => rxvalid(1),
      rxelecidle1 => rxelecidle(1),
      rxstatus1 => rxstatus1,

      txdata2	=> txdata2,
      txdatak2 => txdatak2,
      txelecidle2 => txelecidle(2),
      txcompl2 => txcompl(2),
      rxpolarity2 => rxpolarity(2),
      rxdata2 => rxdata2,
      rxdatak2 => rxdatak2,
      rxvalid2 => rxvalid(2),
      rxelecidle2 => rxelecidle(2),
      rxstatus2 => rxstatus2,

      txdata3	=> txdata3,
      txdatak3 => txdatak3,
      txelecidle3 => txelecidle(3),
      txcompl3 => txcompl(3),
      rxpolarity3 => rxpolarity(3),
      rxdata3 => rxdata3,
      rxdatak3 => rxdatak3,
      rxvalid3 => rxvalid(3),
      rxelecidle3 => rxelecidle(3),
      rxstatus3 => rxstatus3,

      cfg_prmcsr => open,
      cfg_devcsr => open,
      cfg_linkcsr => cfg_linkcsr,
      cfg_ltssm => open,
      slv_dataout => slv_dataout,
      slv_bytevalid => slv_bytevalid,
      slv_bytecount => slv_bytecount,
      slv_dwcount => slv_dwcount,
      slv_addr => slv_addr,
      slv_bar => slv_bar,
      slv_readreq => slv_readreq,
      slv_cpladdr => slv_cpladdr,
      slv_cplparam => slv_cplparam,
      slv_writereq => slv_writereq,
      slv_write => slv_write,
      slv_lastwrite => slv_lastwrite,
      slv_accept => slv_accept,
      slv_abort => slv_abort,
      dma_rd => dma_rd,
      dma_rdaddr => dma_rdaddr,
      dma_rdchannel => dma_rdchannel,
      dma_rddata => dma_rddata,
      dma_wr => dma_wr,
      dma_wraddr => dma_wraddr,
      dma_wrchannel => dma_wrchannel,
      dma_wrdata => dma_wrdata,
      dma_wrbytevalid => dma_wrbytevalid,
      dma0_regin => dma0_regin,
      dma0_regout => dma0_regout,
      dma0_param => dma0_param,
      dma0_control => dma0_control,
      dma0_status => dma0_status,
      dma0_fifocnt => dma0_fifocnt,
      dma1_regin => dma1_regin,
      dma1_regout => dma1_regout,
      dma1_param => dma1_param,
      dma1_control => dma1_control,
      dma1_status => dma1_status,
      dma1_fifocnt => dma1_fifocnt,
      dma2_regin => dma2_regin,
      dma2_regout => dma2_regout,
      dma2_param => dma2_param,
      dma2_control => dma2_control,
      dma2_status => dma2_status,
      dma2_fifocnt => dma2_fifocnt,
      dma3_regin => dma3_regin,
      dma3_regout => dma3_regout,
      dma3_param => dma3_param,
      dma3_control => dma3_control,
      dma3_status  => dma3_status,
      dma3_fifocnt  => dma3_fifocnt
      );


  -- LED shows number of configured lanes.
  usr_led <= cfg_linkcsr(23 downto 20);

  -- assigment of slave transaction abort/accept
  slv_accept <= res_mgt_slv_accept or sram_slv_accept;
  slv_abort <='0';

  -- multiplex DMA data to be read coming from CPL or DMA
  dma_rddata <= res_dma_rddata when dma_rdchannel(2)='1'    -- channel2: CPL (read) for ressource registers
        else fifo_rd_dma_rddata;                            -- channel0: DMA FIFO read transfer

  -- nothing to multiplex: dma write channel is only used by FIFO
  fifo_wr_dma_wrdata <= dma_wrdata;

  swap_sram <= swap_sram_reg;
  
  -- ressource always accepts write transaction or read transaction if dma
  -- channel is free to use for CPL 
  cpl_readreq2 <= '1' when slv_readreq='1' and dma2_status(3)='0' else '0';
  res_mgt_slv_accept <= '1' when slv_bar(0)='1' and (slv_writereq='1' or cpl_readreq2='1') else '0';
  
  --------------------------------------------------------------------------
  -- KFCR Board Info Register (30h)
  -- 
  -- (31:28) product ID    0x1:GRAPE-7 0x2:GRAPE-7E 0x3:GRAPE-DR
  -- (27:14) == backend dependent ==
  -- (13:12) piow sram size  0x0:0.25k 0x1:0.5k 0x2:1k 0x3:2k in 64-bit words
  -- (11:10) fo fifo size    0x0:0.5k 0x1:1k 0x2:2k 0x3:4k in 64-bit words
  -- (9:0)   == backend dependent ==
  -- 
  --------------------------------------------------------------------------
  
  process (clk,rstn)
  begin
    if rstn='0' then
      -- set to default value when reset
      board_info_reg(31 downto 14) <= board_info(31 downto 14);
      board_info_reg(13 downto 12) <= conv_std_logic_vector(SRAM_ADDR_WIDTH-8, 2);
      board_info_reg(11 downto 10) <= conv_std_logic_vector(FIFO_SIZE-9, 2);
      board_info_reg(9 downto 0) <= board_info(9 downto 0);
    elsif rising_edge (clk) then
      if slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0110"
        and slv_bytevalid(3 downto 0)="1111" then
        board_info_reg <= slv_dataout(31 downto 0);
      end if;
    end if;
  end process; 

  --------------------------------------------------------------------------
  -- Register Assignment when a write transaction is targeting BAR0 
  --------------------------------------------------------------------------
  -- 00h : DMA0 address LSB register
  -- 04h : DMA0 address MSB register
  -- 08h : DMA0 size register
  -- 0ch : DMA0 command register
  -- 10h : DMA1 address LSB register
  -- 14h : DMA1 address MSB register
  -- 18h : DMA1 size register
  -- 1ch : DMA1 command register
  -- 20h : Main status register
  -- 24h : Mailbox register
  --       bit (3:0) controls LEDs.
  --       bit (1:0) indicates board function modes.
  --            00: normal mode. dma_wrdata is passed to the backend.
  --            01: dma_wrdata is looped back to internal FIFO.
  --            10: internal FIFO is bypassed (received data is abandoned).
  --            (mode 01 and 10 are just for performance measurement)
  --       bit (2) 1: denotes host finished piow burst,
  --                  and now safe to swap sram0 & 1
  -- 28h : DMA flags and FIFO counters
  -- 2ch..30h : reserved
  -- 30h : KFCR board info register
  -- 34h : Interrupt register
  -- 38h..7ch : reserved
  --------------------------------------------------------------------------
  process (clk,rstn)
  begin
    if rstn='0' then
      res_mailbox_reg <=(others=>'0');
--      res_int_reg <='0'; 
      dma_reset <='0';
      swap_sram_reg <= '0';
    elsif rising_edge (clk) then
      -- 20h : Main status register
      -- Writing a '1' to bit 31 of main status register resets DMA and FIFO
      if slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0100"
        and slv_bytevalid(3)='1' and slv_dataout(31)='1' then
        dma_reset <='1';
      else                                
        dma_reset <='0';
      end if;
      -- Writing a '1' to bit 30 of main status register resets backend pipeline
      if slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0100"
        and slv_bytevalid(3)='1' and slv_dataout(30)='1' then
        reset_backend_reg <='1';
      else                
        reset_backend_reg <='0';
      end if;
      
      -- 24h: Mailbox register 
      if slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0100"
        and slv_bytevalid(7 downto 4)/="0000" then
        res_mailbox_reg <=slv_dataout(63 downto 32);                    
      end if;        

      -- 28h: EHIB command & status register
      -- 
      -- 63    60 59    56 55    52 51    48 47    44 43    40 39    36 35    32
      --   0000     0000     0000     0000     0adf     ffff     ffff     ffff
      --                                        ^
      --                                        |^
      --                                        ||^     ^^^^     ^^^^     ^^^^
      --                                        |||
      --                                        ||+-- dma1_fifocnt(12 downto 0) (r)
      --                                        |+--- dma1_done (r)
      --                                        +---- sram1_wlock (r)
      -- 
      -- 31    28 27    24 23    20 19    16 15    12 11     8  7     4  3    0
      --   0000     000w     wwww     wwww     sadf     ffff     ffff     ffff
      --               ^     ^^^^     ^^^^     ^ 
      --               |                       |^
      --               |                       ||^
      --               |                       |||^     ^^^^     ^^^^     ^^^^
      --               |                       ||||
      --               |                       |||+-- dma0_fifocnt(12 downto 0) (r)
      --               |                       ||+--- dma0_done (r)
      --               |                       |+---- sram0_wlock (r)
      --               |                       +----- swap_sram_reg (r/w)
      --               +----------------------------- sram_wcnt(8 downto 0) (r/w)
      -- 

      -- swap_sram_reg
      -- asserted by piow. deasserted when swap_done.
      if swap_done='1' then
        swap_sram_reg <= '0';
      elsif slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0101"
        and slv_bytevalid(1)='1' then
        swap_sram_reg <= slv_dataout(15);
      end if;        

      -- 34h: Interrupt register
      -- when an interrupt is received from DMA management block
      -- PCI-Ez backend signal int_request is asserted
      -- it will be de-asserted when writing one at bit 0 of interrupt reg
--      if slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0110"
--         and slv_bytevalid(4)='1' and slv_dataout(32)='1' then
--         res_int_reg <='0';
--       elsif dma_int_req='1' then
--         res_int_reg <='1';
--       end if;
    end if;
  end process;        
  
  dma_test_mode <= res_mailbox_reg(1 downto 0);
--  int_request <= res_int_reg;                
  
  -- MAIN STATUS REGISTER assignment (READ ONLY Register)
  -- (report that Ref design version is 1.2)
  res_main_status_reg <=x"00000012";
  
  --------------------------------------------------------------------------
  -- Prepare a completion when a read transaction is targeting BAR0 
  --------------------------------------------------------------------------        
  
  -- save necessary registers in order to make the completion                        
  dma2_regin(127 downto 96) <= x"000000" & '0' & slv_addr(6 downto 0); --local address    
  dma2_regin(76 downto 64) <= slv_bytecount;
  dma2_regin(31 downto 0) <= slv_cpladdr;
  dma2_param(15 downto 11) <= slv_cplparam; 
  
  -- for completion, PCI address MSB, upper transfert size registers are set to zero
  dma2_regin(63 downto 32) <=(others=>'0');
  dma2_regin(95 downto 77) <=(others=>'0');          
  
  -- completion parameter:   CPL      BE      RL    Res   DMA(Ram Mode)
  dma2_param(10 downto 0) <= "100" & "0000" & "00" & '0' & '1';
  
  dma2_control <= "011111" when slv_bar(0)='1' and cpl_readreq2='1' else (others=>'0'); 
  
  -- fifo count is unused for completion because using DMA Mode as RAM mode
  dma2_fifocnt <=(others=>'0');
  
  -- assign data when dma is currently reading                        
  process (clk,rstn)
  begin
    if rstn='0' then
      res_dma_rddata <= (others=>'0');
    elsif rising_edge (clk) then
      if dma_rd='1' then
        case dma_rdaddr(6 downto 3) is                          
          when "0000" =>        -- 00h: DMA0 address register  
            res_dma_rddata <= dma0_regout(63 downto 0);                                                
          when "0001" =>        -- 08h: DMA0 command-size register (and DMA0 status register)
            res_dma_rddata <=  x"000" & dma0_status & x"00E0" & dma0_regout(95 downto 64);
          when "0010" =>         -- 10h: DMA1 address register
            res_dma_rddata <= dma1_regout(63 downto 0);
          when "0011" =>        -- 18h: DMA1 command-size register (and DMA0 status register) 
            res_dma_rddata <= x"000" & dma1_status & x"00F0" & dma1_regout(95 downto 64);                
          when "0100" =>        -- 20h: Mailbox register & Main status register
            res_dma_rddata <= res_mailbox_reg & res_main_status_reg;

          when "0101" =>    -- 28h: a PIO w flag, DMA r/w flags, and FIFO counters
            res_dma_rddata <= conv_std_logic_vector(0, 32-2-13)
                              & sram1_wlock & dma1_done & dma1_fifocnt
                              & conv_std_logic_vector(0, 32-SRAM_ADDR_WIDTH-3-13)
                              & sram_wcnt & swap_sram_reg & sram0_wlock & dma0_done & dma0_fifocnt;
          when "0110" =>        -- 30h: KFCR board info register
            res_dma_rddata <= x"00000000" & board_info_reg;
          when others =>
            res_dma_rddata <= (others=>'0');
        end case;         
      end if;
    end if;
  end process;

  -- define read/write request from/to backend/fifo

  hib_we <= '1' when dma_wr='1' and dma_wrchannel(0)='1' and dma_test_mode="00" -- DMA read
            else sram_oe when dma_test_mode="00" -- PIO write
            else '0';

  hib_data <= fifo_wr_dma_wrdata when dma_wr='1' and dma_wrchannel(0)='1'
              else sram_rdata;

  wr_fiforeq <= '1' when dma_wr='1' and dma_wrchannel(0)='1' and dma_test_mode="01"
                else sram_oe when dma_test_mode="01"
                else backend_we;

  wr_fifodata <= fifo_wr_dma_wrdata when dma_wr='1' and dma_wrchannel(0)='1' and dma_test_mode="01"
                 else sram_rdata when dma_test_mode="01"
                 else backend_data;

  rd_fiforeq <= '1' when dma_rd='1' and dma_rdchannel(1)='1' else '0';

  fifo : hibscfifo_sgx2
    generic map(
      ADDR_WIDTH         => FIFO_SIZE,
      DATA_WIDTH         => 64
      )
    port map(
      show_ahead        => '0',
      clk               => clk,
      rstn              => rstn,
      sclr              => dma_reset,
      wrreq             => wr_fiforeq,
      wrdata            => wr_fifodata,
      wrfull            => open,
      wrempty           => open,
      wrusedw           => open,
      rdreq             => rd_fiforeq,
      rddata            => fifo_rd_dma_rddata,
      rdempty           => open,
      rdfull            => open,
      rdusedw           => rd_fifosize
      );

  -- dma0 control management
  dma0_control(0) <= '1' when slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0000" and slv_bytevalid(0)='1' else '0';
  dma0_control(1) <= '1' when slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0000" and slv_bytevalid(4)='1' else '0';
  dma0_control(2) <= '1' when slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0001" and slv_bytevalid(0)='1' else '0';
  dma0_control(3) <= '0';        -- local address not used
  dma0_control4 <= '1' when slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0001" and slv_bytevalid(4)='1' else '0';
  dma0_control(4) <= dma0_control4;
  dma0_control(5) <= dma_reset; -- stop dma when FIFO is flushed

  -- dma0 control management
  dma1_control(0) <= '1' when slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0010" and slv_bytevalid(0)='1' else '0';
  dma1_control(1) <= '1' when slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0010" and slv_bytevalid(4)='1' else '0';
  dma1_control(2) <= '1' when slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0011" and slv_bytevalid(0)='1' else '0';
  dma1_control(3) <= '0';        -- local address not used
  dma1_control4 <= '1' when slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0011" and slv_bytevalid(4)='1' else '0';
  dma1_control(4) <= dma1_control4;
  dma1_control(5) <= dma_reset; -- stop dma when FIFO is flushed

  --------------------------------------------------------------------------
  -- DMA read (Host -> EHIB)
  --------------------------------------------------------------------------

  dma0_regin(127 downto 0) <= slv_dataout & slv_dataout;
  dma0_param <= "00" & "000" & "110" & "0000" & "0000";                 -- ATTR,TC,MEM RD,BE,FIFO latency, FIFO mode fixed
  dma0_fifocnt <="1" & conv_std_logic_vector(0, 12);  -- backend is always ready to receive data

  --------------------------------------------------------------------------
  -- DMA write (Host <- EHIB)
  --------------------------------------------------------------------------
  
  dma1_regin(127 downto 0) <= slv_dataout & slv_dataout;
  dma1_param <= "00" & "000" & "111" & "0000" & "0000";                -- ATTR,TC,MEM WR,BE,FIFO latency, FIFO mode fixed

  -- Indicate how many bytes are stored in FIFO
  -- dma1_fifocnt: byte count
  -- rd_fifosize: 64-bit word count
  dma1_fifocnt <="1000000000000" when dma_test_mode="10" else
                  conv_std_logic_vector(0, 10-FIFO_SIZE) & rd_fifosize & "000";
  
  -- dmaN_end
  -- detect a falling edge dmaN_status(3)
  -- that indicate an end of DMA transfer        
  process (clk,rstn)
  begin
    if rstn='0' then
      dma0_end <= '0';
      dma0_status3_r <= '0'; 
    elsif rising_edge (clk) then    
      dma0_status3_r <= dma0_status(3);
      if dma0_status(3)='0' and dma0_status3_r='1' then
        dma0_end <= '1';
      else
        dma0_end <= '0';
      end if;
    end if;
  end process;

  process (clk,rstn)
  begin
    if rstn='0' then
      dma1_end <= '0';
      dma1_status3_r <= '0'; 
    elsif rising_edge (clk) then    
      dma1_status3_r <= dma1_status(3);
      if dma1_status(3)='0' and dma1_status3_r='1' then
        dma1_end <= '1';
      else
        dma1_end <= '0';
      end if;
    end if;
  end process;

  -- dmaN_done
  -- value 1 indicates DMA-transfer completion.
  -- the bit is cleared at start of a new transaction,
  -- which is signaled by dmaN_control(4)
  process (clk,rstn)
  begin
    if rstn='0' then
      dma0_done <= '0';
    elsif rising_edge (clk) then    
      if dma0_control4 = '1' then
        dma0_done <= '0';
      elsif dma0_end = '1' then
        dma0_done <= '1';
      end if;
    end if;
  end process;
  
  process (clk,rstn)
  begin
    if rstn='0' then
      dma1_done <= '0';
    elsif rising_edge (clk) then
      if dma1_control4 = '1' then
        dma1_done <= '0';
      elsif dma1_end ='1' then
        dma1_done <= '1';
      end if;
    end if;
  end process;
  
  -- End of DMA generate an Interrupt
--  dma_int_req <= dma1_end;
--  dma_int_req <= '0'; -- don't use interrupt

-- 
-- sram0 and sram1:
-- double SRAM buffers to receive data from write-combining buffer
-- of x86 CPU of the host. double buffer is necessary to parallelize
-- transfer A and B.
-- 
--                   A
-- host       sram0 ----> backend
--      ----> sram1
--        B
-- 
-- note that data may not comes in address
-- order, and thus FIFO is not a practical choice.
-- 
  sram0 : hibdcrambe_sgx2
    generic map (
      ADDR_WIDTH    => SRAM_ADDR_WIDTH,
      DATA_WIDTH    => 64,
      BYTE_SIZE     => 8
      )
    port map (
      wrclk         => clk,
      wraddr        => sram0_waddr,
      wrdata        => sram0_wdata,    
      wrbe          => sram0_we,
      rdclk         => clk,
      rdaddr        => sram0_raddr,
      rddata        => sram0_rdata
      );

  sram1 : hibdcrambe_sgx2
    generic map (
      ADDR_WIDTH    => SRAM_ADDR_WIDTH,
      DATA_WIDTH    => 64,
      BYTE_SIZE     => 8
      )
    port map (
      wrclk         => clk,
      wraddr        => sram1_waddr,
      wrdata        => sram1_wdata,    
      wrbe          => sram1_we,
      rdclk         => clk,
      rdaddr        => sram1_raddr,
      rddata        => sram1_rdata
      );      

  -- either one of sram0 or sram1 is ready to be written.
  sram0_we <= slv_bytevalid when slv_write='1' and slv_bar(2)='1'
              and sram0_wlock_reg='0' else (others => '0');
  sram1_we <= slv_bytevalid when slv_write='1' and slv_bar(2)='1'
              and sram0_wlock_reg='1' and sram1_wlock_reg='0' else (others => '0');

  -- s_addr: byte address  sramN_waddr: 8-byte word address
  -- s_addr(2..0) are discarded.
  sram0_waddr <= slv_addr(SRAM_ADDR_WIDTH-1+3 downto 3);
  sram1_waddr <= slv_addr(SRAM_ADDR_WIDTH-1+3 downto 3);

  sram0_wdata <= slv_dataout;
  sram1_wdata <= slv_dataout;
  
  sram0_raddr <= sram0_rcnt_up;
  sram1_raddr <= sram1_rcnt_up;
  
  -- sram0/1_oe
  process (clk,rstn)
  begin
    if rstn='0' then
      sram0_oe <= '0';
    elsif rising_edge (clk) then
      if sram0_rcnt_dn /= conv_std_logic_vector(0,SRAM_ADDR_WIDTH) then
        sram0_oe <= '1';
      else
        sram0_oe <= '0';
      end if;
    end if;
  end process;

  process (clk,rstn)
  begin
    if rstn='0' then
      sram1_oe <= '0';
    elsif rising_edge (clk) then
      if sram1_rcnt_dn /= conv_std_logic_vector(0,SRAM_ADDR_WIDTH) then
        sram1_oe <= '1';
      else
        sram1_oe <= '0';
      end if;
    end if;
  end process;

  process (clk,rstn)
  begin
    if rstn='0' then
      sram_oe <= '0';
    elsif rising_edge (clk) then
      sram_oe <= sram0_oe or sram1_oe;
    end if;
  end process;

  sram_rdata <= sram0_rdata when sram0_wlock_reg='1'
                else sram1_rdata when sram1_wlock_reg='1'
                else (others => '0');
  
  -- sram0/1_wlock
  process (clk,rstn)
  begin
    if rstn='0' then
      sram0_wlock_reg <= '0';
    elsif rising_edge (clk) then
      if swap_sram='1' and swap_done='0' then
        if sram0_wlock_reg='0' and sram1_oe='0' then
          sram0_wlock_reg <= '1';
          swap_done <= '1';
        elsif sram1_wlock_reg='0' and sram0_oe='0' then
          sram0_wlock_reg <= '0';
          swap_done <= '1';
        end if;
      else
        swap_done <= '0';
      end if;
    end if;
  end process;

  sram1_wlock_reg <= '1' when sram0_wlock_reg='0' else '0';
  sram0_wlock <= sram0_wlock_reg;
  sram1_wlock <= sram1_wlock_reg;
  
  -- sram_wcnt
  -- data size written to sram0/1 is given by the host

  process (clk,rstn)
  begin
    if rstn='0' then
      sram_wcnt <= (others => '0');
    elsif rising_edge (clk) then
      if slv_bar(0)='1' and slv_write='1' and slv_addr(6 downto 3)="0101"
        and slv_bytevalid(3 downto 2)="11" then
        sram_wcnt <= slv_dataout(SRAM_ADDR_WIDTH-1+16 downto 16);  -- !!
      end if;
    end if;
  end process;

  -- sram0_rcnt
  process (clk,rstn)
  begin
    if rstn='0' then
      sram0_rcnt_dn <= (others => '0');
      sram0_rcnt_up <= (others => '0');
    elsif rising_edge (clk) then
      if sram0_wlock_reg='0' then
        sram0_rcnt_dn <= (others => '0');
        sram0_rcnt_up <= (others => '0');
      elsif sram0_wlockD='0' then -- rising edge of sram0_wlock
        sram0_rcnt_dn <= sram_wcnt;
      elsif sram0_rcnt_dn /= conv_std_logic_vector(0,SRAM_ADDR_WIDTH) then
        sram0_rcnt_dn <= sram0_rcnt_dn - '1';
        sram0_rcnt_up <= sram0_rcnt_up + '1';
      end if;
    end if;
  end process;

  -- sram1_rcnt
  process (clk,rstn)
  begin
    if rstn='0' then
      sram1_rcnt_dn <= (others => '0');
      sram1_rcnt_up <= (others => '0');
    elsif rising_edge (clk) then
      if sram1_wlock_reg='0' then
        sram1_rcnt_dn <= (others => '0');
        sram1_rcnt_up <= (others => '0');
      elsif sram1_wlockD='0' then -- rising edge of sram1_wlock
        sram1_rcnt_dn <= sram_wcnt;
      elsif sram1_rcnt_dn /= conv_std_logic_vector(0,SRAM_ADDR_WIDTH) then
        sram1_rcnt_dn <= sram1_rcnt_dn - '1';
        sram1_rcnt_up <= sram1_rcnt_up + '1';
      end if;
    end if;
  end process;

  -- sram0_wlockD
  process (clk,rstn)
  begin
    if rstn='0' then
      sram0_wlockD <= '0';
    elsif rising_edge (clk) then
      sram0_wlockD <= sram0_wlock_reg;
    end if;
  end process;

  -- sram1_wlockD
  process (clk,rstn)
  begin
    if rstn='0' then
      sram1_wlockD <= '0';
    elsif rising_edge (clk) then
      sram1_wlockD <= sram1_wlock_reg;
    end if;
  end process;
  
  -- sramN accepts write transaction if sram is ready.
  -- sramN accepts read transaction if dma channel3 is free to use for CPL.
  cpl_readreq3 <= '1' when slv_readreq='1' and dma3_status(3)='0' else '0';
  sram_slv_accept <= '1' when slv_bar(2)='1'
                   and ((slv_writereq='1' and swap_sram='0') or cpl_readreq3='1')
                   else '0';

  
  -- save necessary registers in order to make the completion
  dma3_regin(127 downto 96) <= "0000000000000000000000" & slv_addr(9 downto 0); --local address
  dma3_regin(76 downto 64) <= slv_bytecount;
  dma3_regin(31 downto 0) <= slv_cpladdr;
  dma3_param(15 downto 11) <= slv_cplparam;     
  
  -- for completion, PCI address MSB, upper transfert size registers are set to zero
  dma3_regin(63 downto 32) <= (others => '0');
  dma3_regin(95 downto 77) <= (others => '0');  
  
  -- completion parameter:   CPL      BE      RL    Res   DMA(Ram Mode)
  dma3_param(10 downto 0) <= "100" & "0000" & "01" & '0' & '1';
  
  dma3_control <= "011111" when slv_bar(2)='1' and cpl_readreq3='1'
                  else (others=>'0'); 
  
  -- fifo count is unused for completion becaase using DMA Mode as RAM mode
  dma3_fifocnt <=(others=>'0');
  
end structural;
