---------------------------------------------------------------
--
-- hibscfifo_c2
--
-- General purpose synchronous Single-clock FIFO 
--
-- (c) PLD Applications 2003-2004
-- 
-- v 1.5
--
-- 30-Jun-2003 - ALO+PLE
--  7-jan-2004 - PLE : Added synchronous clear port
--  9-mar-2004 - PLE : Fixed read error in sync mode 
-- 14-may-2004 - PLE : Removed reset pin for DPRAM
-- 
--------------------------------------------------------------------
--
-- Features:
--
-- - Data width and fifo depth is programmable, but one word
--   is lost in order to get better performance and less logic.
-- - show-ahead : 'rdreq' correspond to a read acknowledge 
--   and next data will be available on output data bus on 
--   clock cycle - easy to control but less performance after 
--   synthesis
-- - usedw, empty & full flags are cycle accurate
--
-- Input port "show_ahead" should  be tied to a fixed value :
--   0 : synchronous mode
--   1 : show-ahead mode
--
---------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all; 

entity hibscfifo_c2 is
  generic
    (
      ADDR_WIDTH    : integer:=8;
      DATA_WIDTH    : integer:=32
      );
  port
    (
      show_ahead    : in std_logic;
      
      clk            : in std_logic;
      rstn        : in std_logic;
      sclr        : in std_logic;
      
      wrreq        : in std_logic;
      wrdata        : in std_logic_vector (DATA_WIDTH-1 downto 0);
      wrfull        : out std_logic;
      wrempty        : out std_logic;
      wrusedw         : out std_logic_vector(ADDR_WIDTH-1 downto 0);
      
      rdreq        : in std_logic;
      rddata        : out std_logic_vector (DATA_WIDTH-1 downto 0);
      rdfull        : out std_logic;
      rdempty        : out std_logic;
      rdusedw         : out std_logic_vector(ADDR_WIDTH-1 downto 0)
      );
end;

architecture structural of hibscfifo_c2 is
  
  component hibdcram_c2
    generic(
      ADDR_WIDTH : INTEGER := 8;
      DATA_WIDTH : INTEGER := 32);
    port(
      wrclk : in std_logic;
      wren : in std_logic;
      wraddr : in std_logic_vector((ADDR_WIDTH-1) downto 0);
      wrdata : in std_logic_vector((DATA_WIDTH-1) downto 0);
      rdclk : in std_logic;
      rdaddr : in std_logic_vector((ADDR_WIDTH-1) downto 0);
      rddata : out std_logic_vector((DATA_WIDTH-1) downto 0));
  end component;    
  
  signal rddata_n                        : std_logic_vector (DATA_WIDTH-1 downto 0);
  signal rdad,wrad                    : std_logic_vector (ADDR_WIDTH-1 downto 0);
  signal rdaddr_next                    : std_logic_vector (ADDR_WIDTH-1 downto 0);
  
  signal read_allow,read_allow_sa        : std_logic;
  signal write_allow                    : std_logic;
  signal write_allow_r,write_allow_rr    : std_logic;
  signal rdcount,wrcount                : std_logic_vector (ADDR_WIDTH-1 downto 0);
  signal rdfull_r,rdempty_r            : std_logic;
  signal wrfull_r,wrempty_r            : std_logic; 
  
  constant C0 : std_logic_vector (ADDR_WIDTH-1 downto 0):=(others=>'0'); 
  constant C1 : std_logic_vector (ADDR_WIDTH-1 downto 0):=(others=>'1');

begin
  ----------------------------------------------------------------
  --                                                            --
  --  Dual-port RAM instantiation                               --
  --                                                            --
  ---------------------------------------------------------------- 
  
  dcram : hibdcram_c2
    generic map(ADDR_WIDTH=>ADDR_WIDTH,
                DATA_WIDTH=>DATA_WIDTH) 
    port map(wrclk    => clk,
             wren    => write_allow,
             wraddr    => wrad,
             wrdata    => wrdata,
             rdclk    => clk,
             rdaddr    => rdaddr_next,
             rddata    => rddata_n);
  
  -- Register output
  process (clk,rstn)
  begin
    if rstn='0' then
      rddata <=(others=>'0');    
    elsif rising_edge (clk) then
      if read_allow_sa='1' then
        rddata <=rddata_n;    
      end if;
    end if;
  end process;
  
  
  ----------------------------------------------------------------
  --                                                            --
  -- Address pointers                                              --
  --                                                            --
  ---------------------------------------------------------------- 
  
  process(clk,rstn)
  begin
    if rstn='0' then
      wrad <=(others=>'0');
      rdad <=(others=>'0');
    elsif rising_edge (clk) then
      if sclr='1' then
        wrad <=(others=>'0');
        rdad <=(others=>'0');
      else
        wrad <=unsigned (wrad)+write_allow;    
        
        -- Do not increment read address if FIFO is going to be empty 
        if show_ahead='0' or not (wrcount(ADDR_WIDTH-1 downto 1)=C0(ADDR_WIDTH-1 downto 1)
                                  and wrreq='0' and (wrcount(0)='0' or rdreq='1')) then
          rdad <=rdaddr_next;    
        end if;
      end if;
    end if;
  end process; 
  
  rdaddr_next <=unsigned (rdad)+read_allow_sa;
  
  
  ----------------------------------------------------------------
  --                                                            --
  --  Read/write data counters                                  --
  --                                                            --
  ---------------------------------------------------------------- 

  write_allow <=wrreq and not wrfull_r;
  read_allow <=rdreq and not rdempty_r;
  
  read_allow_sa <=read_allow when show_ahead='0'
                   else '1' when rdreq='1' and rdempty_r='0'
                   else '1' when rdempty_r='1' and write_allow_rr='1'
                   else '0';
  
  -- Data counters
  process (clk,rstn)
  begin
    if rstn='0' then
      write_allow_r <='0';
      write_allow_rr <='0';
      wrcount <=(others=>'0');
      rdcount <=(others=>'0');
    elsif rising_edge (clk) then
      if sclr='1' then
        write_allow_r <='0';
        write_allow_rr <='0';
        wrcount <=(others=>'0');
        rdcount <=(others=>'0');
      else
        write_allow_r <=write_allow;
        write_allow_rr <=write_allow_r;
        
        if read_allow='1' and write_allow='0' then
          wrcount <=unsigned (wrcount)-'1';
        elsif read_allow='0' and write_allow='1' then
          wrcount <=unsigned (wrcount)+'1';
        end if;
        
        if read_allow='1' and write_allow_rr='0' then
          rdcount <=unsigned (rdcount)-'1';
        elsif read_allow='0' and write_allow_rr='1' then
          rdcount <=unsigned (rdcount)+'1';
        end if;
      end if;
    end if;
  end process;
  
  wrusedw <=wrcount;
  rdusedw <=rdcount;
  
  ----------------------------------------------------------------
  --                                                            --
  --  Empty/full flags                                          --
  --                                                            --
  ---------------------------------------------------------------- 
  
  process (clk,rstn)
  begin
    if rstn='0' then
      wrfull_r <='0';
      wrempty_r <='1';
      rdfull_r <='0';
      rdempty_r <='1';
    elsif rising_edge (clk) then
      if sclr='1' then
        wrfull_r <='0';
        wrempty_r <='1';
        rdfull_r <='0';
        rdempty_r <='1';
      else
        if wrcount(ADDR_WIDTH-1 downto 1)=C0(ADDR_WIDTH-1 downto 1) and wrreq='0' and
          (wrcount(0)='0' or rdreq='1') then
          wrempty_r <= '1';
        else
          wrempty_r <= '0';
        end if;
    
        if wrcount(ADDR_WIDTH-1 downto 1)=C1(ADDR_WIDTH-1 downto 1) and rdreq='0' and
          (wrcount(0)='1' or wrreq='1') then
          wrfull_r <= '1';
        else
          wrfull_r <= '0';
        end if;
        
        if rdcount(ADDR_WIDTH-1 downto 1)=C0(ADDR_WIDTH-1 downto 1) and write_allow_rr='0' and
          (rdcount(0)='0' or rdreq='1') then
          rdempty_r <= '1';
        else
          rdempty_r <= '0';
        end if;

        if rdcount(ADDR_WIDTH-1 downto 1)=C1(ADDR_WIDTH-1 downto 1) and rdreq='0' and
          (rdcount(0)='1' or write_allow_rr='1') then
          rdfull_r <= '1';
        else
          rdfull_r <= '0';
        end if;
      end if;
    end if;
  end process;
  
  wrempty <=wrempty_r;
  wrfull <=wrfull_r; 
  rdempty <=rdempty_r;
  rdfull <=rdfull_r; 
  
end structural;
