library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity ifpga is 
  port (
    pci_clk                : in    std_logic;
    pci_rst                : in    std_logic;
    pci_ad                 : inout std_logic_vector(63 downto 0);
    pci_cbe                : inout std_logic_vector(7 downto 0);
    pci_frame              : inout std_logic;
    pci_irdy               : inout std_logic;
    pci_trdy               : inout std_logic;
    pci_stop               : inout std_logic;
    pci_devsel             : inout std_logic;
    pci_idsel              : in    std_logic;
    pci_req64              : inout std_logic;
    pci_ack64              : inout std_logic;
    pci_par                : inout std_logic;
    pci_par64              : inout std_logic;
    pci_perr               : inout std_logic;
    pci_serr               : inout std_logic;
    pci_inta               : inout std_logic;
    pci_req                : inout std_logic;
    pci_gnt                : in    std_logic;
    pci_m66en              : in    std_logic;
    pci_intb               : inout std_logic;
    pci_intc               : inout std_logic;
    pci_intd               : inout std_logic;
    pci_pme                : inout std_logic;
    pci_lock               : inout std_logic;

    o_sw                   : in std_logic_vector(1 downto 0);
    o_led                  : out std_logic_vector(3 downto 0);

    l_eclk                 : out std_logic;
    l_iclk                 : in std_logic
    );
end ifpga;

architecture hierarchy of ifpga is

  component hib
    port (
      -- PCI-X signals
      clk_pci             : in    std_logic;
      rst_pci             : in    std_logic;
      idsel_pci           : in    std_logic;
      lock_pci            : in    std_logic;
      m66en_pci           : in    std_logic;
      ad_pci              : inout std_logic_vector(63 downto 0);
      frame_pci           : inout std_logic;
      req64_pci           : inout std_logic;
      cbe_pci             : inout std_logic_vector(7 downto 0);
      irdy_pci            : inout std_logic;
      req_pci             : inout std_logic;
      gnt_pci             : in    std_logic;
      trdy_pci            : inout std_logic;
      stop_pci            : inout std_logic;
      devsel_pci          : inout std_logic;
      ack64_pci           : inout std_logic;
      par_pci             : inout std_logic;
      par64_pci           : inout std_logic;
      perr_pci            : inout std_logic;
      serr_pci            : inout std_logic;
      inta_pci            : inout std_logic;

      -- User Leds
      usr_led             : out std_logic_vector (7 downto 0);
      
      -- PLL reconfiguration control signals
      cg_wrreq            : out std_logic;
      cg_data             : out std_logic_vector(24 downto 0);
      cg_busy             : in  std_logic;
      
      -- PLDA PCI-X board specific signals
      prot0_out           : out    std_logic;
      prot1_out           : out    std_logic;
      prot1_in            : in    std_logic_vector(1 downto 0);

      -- interface to the backend 
      hib_we              : out std_logic;
      hib_data            : out std_logic_vector(63 downto 0);
      backend_we          : in std_logic;
      backend_data        : in std_logic_vector(63 downto 0);
      reset_backend       : out std_logic;

      -- board information register initial value
      board_info          : in std_logic_vector(31 downto 0)
      );
  end component;

  component backend
    port (
      hib_clk              : in std_logic;
      rst                  : in std_logic; -- negative logic

      -- board info register initial value
      board_info         : out std_logic_vector(31 downto 0);

      -- led output
--      o_led              : out std_logic_vector(3 downto 0);

      -- DMA signals
      hib_we               : in std_logic;
      hib_data             : in std_logic_vector(63 downto 0);
      backend_we           : out std_logic;
      backend_data         : out std_logic_vector(63 downto 0)
      );
  end component;

  component cg
    port (
      -- a reference clock externally provided.
      ext_clk            : in std_logic;
      
      -- clock signals this module provides.
      clk0               : out std_logic;  -- interface clk (non programmable)
      clk1               : out std_logic;  -- backend clk (programmable)

      -- PLL reconfiguration control signals.
      reset              : in  std_logic;  -- reset this module. active high.
      wrreq              : in  std_logic;  -- pll configuration parameter write enable.
      datain             : in  std_logic_vector(24 downto 0); -- pll configuration parameter.
      busy               : out std_logic  -- pll configuration is in progress.
      );
  end component;

  signal hib_clk        : std_logic;
  signal backend_clk0   : std_logic;
  signal reset_backend  : std_logic;
  signal board_info     : std_logic_vector(31 downto 0);
  signal hib_we         : std_logic;
  signal hib_data       : std_logic_vector(63 downto 0);
  signal hib_led        : std_logic_vector(7 downto 0);
  signal backend_we     : std_logic;
  signal backend_data   : std_logic_vector(63 downto 0);
  signal backend_led    : std_logic_vector(3 downto 0);
  signal cg_wrreq       : std_logic;
  signal cg_data        : std_logic_vector(24 downto 0);
  signal cg_busy        : std_logic;

begin

  cg_instance : cg
    port map (
      ext_clk => pci_clk,
      clk0    => hib_clk,
      clk1    => backend_clk0,
      reset   => not reset_backend,     -- reset:active high    reset_backend:acrive low
      wrreq   => cg_wrreq,
      datain  => cg_data,
      busy    => cg_busy
    );
  
  hib_instance : hib
    port map (
      -- PCI-X signals
      idsel_pci => pci_idsel,
      lock_pci => pci_lock,
      ad_pci => pci_ad,
      frame_pci => pci_frame,
      req64_pci => pci_req64,
      cbe_pci => pci_cbe,  
      irdy_pci => pci_irdy, 
      trdy_pci => pci_trdy, 
      stop_pci => pci_stop, 
      devsel_pci => pci_devsel,
      ack64_pci => pci_ack64,
      par_pci => pci_par, 
      par64_pci => pci_par64,
      perr_pci => pci_perr,
      serr_pci => pci_serr,
      inta_pci => pci_inta,
      req_pci => pci_req, 
      gnt_pci => pci_gnt,
      m66en_pci => pci_m66en,

      prot1_in => "00",
      
      -- global lines
      clk_pci     => hib_clk,
      rst_pci     => pci_rst,
      usr_led     => hib_led,

      -- PLL reconfiguration control signals
      cg_wrreq    => cg_wrreq,
      cg_data     => cg_data,
      cg_busy     => cg_busy,
      
      -- interface to the backend 
      hib_we        => hib_we,
      hib_data      => hib_data,
      backend_we    => backend_we,
      backend_data  => backend_data,
      reset_backend => reset_backend,
      board_info    => board_info
      );

  backend_instance: backend
    port map (
      hib_clk      => hib_clk,
      rst          => reset_backend,    -- active low
      board_info   => board_info,
--      o_led        => backend_led,
      hib_we       => hib_we,
      hib_data     => hib_data,
      backend_we   => backend_we,
      backend_data => backend_data
      );

  -- LED control
--  o_led <= backend_led or hib_led(3 downto 0);
  o_led <= hib_led(3 downto 0);
  
end hierarchy;
