#!/usr/bin/perl -w

=pod

    A script to generate the top-level instance of interface logic
    hierarchy of an given architecture <arch>, so that the backend
    logic <backend> written by the user is wrapped around.

    usage: ifpgagen.pl <arch> <backend>
        arch:
            dr3:     GRAPE-DR TB3 (preliminary)   PCIe x4       125MHz
            g7m1:    KFCR GRAPE-7 model100        PCI-X         133MHz
            g7m8:    KFCR GRAPE-7 model800        PCI-X         133MHz
            g7p1:    KFCR GRAPE-7 model300/600    pFPGA1,4      100MHz
            g7p2:    KFCR GRAPE-7 model300/600    pFPGA2,3,5    100MHz
            g7p6:    KFCR GRAPE-7 model300/600    pFPGA6        100MHz
            dkx4:    PLDA DesignKit               PCIe x4       125MHz
            gx2x4:   PLDA XpressGX2               PCIe x4       125MHz
            gx2x4f:  PLDA XpressGX2               PCIe x4       125MHz, fully licensed core
            gx2x8:   PLDA XpressGX2               PCIe x8       250MHz
            gx2x8f:  PLDA XpressGX2               PCIe x8       250MHz, fully licensed core

        backend: a VHDL file that contains the 'backend' entity.
            samples can be found at $GWRAPPATH/samples/

=cut

    use File::Basename;
    use File::Spec;

# use strict;

# ifpga vhdl template
%ifpgatemplates = (
                   'dkx4'    => 'ifpga_dkx4.vhd',
                   'gx2x4'   => 'ifpga_gx2x4.vhd',
                   'gx2x4f' => 'ifpga_gx2x4f.vhd',
                   'gx2x8'   => 'ifpga_gx2x8.vhd',
                   'gx2x8f'  => 'ifpga_gx2x8f.vhd',
                   'g7m1'    => 'ifpga_g7m1.vhd',
                   'g7m8'    => 'ifpga_g7m8.vhd',
                   'g7p1'    => 'pfpga_g7p1.vhd',
                   'g7p2'    => 'pfpga_g7p2.vhd',
                   'g7p6'    => 'pfpga_g7p6.vhd',
                   'dr3'     => 'ifpga_dr3.vhd',
                   );

# hib backend interface
@hportnames = (
               'hib_we',
               'hib_data',
               'backend_we',
               'backend_data',
               'backend_run',
               'board_info',
               'hib_clk',
               'backend_clk0',
               'backend_clk1',
               'backend_clk2',
               'backend_clk3',
               'backend_clk4',
               'backend_clk5',
               'rst',
               );

if (!defined $ARGV[1]) {
    my $umsg = << 'EOUMSG';

    A script to generate the top-level instance of interface logic
    hierarchy of an given architecture <arch>, so that the backend
    logic <backend> written by the user is wrapped around.

    usage: ifpgagen.pl <arch> <backend>
        arch:
            dr3:     GRAPE-DR TB3 (preliminary)   PCIe x4       125MHz
            g7m1:    KFCR GRAPE-7 model100        PCI-X         133MHz
            g7m8:    KFCR GRAPE-7 model800        PCI-X         133MHz
            g7p1:    KFCR GRAPE-7 model300/600    pFPGA1,4      100MHz
            g7p2:    KFCR GRAPE-7 model300/600    pFPGA2,3,5    100MHz
            g7p6:    KFCR GRAPE-7 model300/600    pFPGA6        100MHz
            dkx4:    PLDA DesignKit               PCIe x4       125MHz
            gx2x4:   PLDA XpressGX2               PCIe x4       125MHz
            gx2x4f:  PLDA XpressGX2               PCIe x4       125MHz, fully licensed core
            gx2x8:   PLDA XpressGX2               PCIe x8       250MHz
            gx2x8f:  PLDA XpressGX2               PCIe x8       250MHz, fully licensed core

        backend: a VHDL file that contains the 'backend' entity.
            samples can be found at $GWRAPPATH/samples/
EOUMSG
    die "$umsg\n";
}

if (!$ifpgatemplates{$ARGV[0]}) {
    die "Invalid interface type: \'$ARGV[0]\'.\n";
}
$ifpgafile = $ifpgatemplates{$ARGV[0]}; # ifpga VHDL source template
$backendfile = $ARGV[1]; # backend VHDL source

if (defined $ENV{GWRAPPATH}) {
    $gwrappath = $ENV{GWRAPPATH};
}
else {
    $gwrappath = '.';
}
my ($name, $path) = fileparse($ifpgafile);
$ifpgafile = File::Spec->catfile($gwrappath, $ifpgafile);

@bport = parseBackend($backendfile);

# check if the backend port should be connected to a hib port,
# or to an external pin.


foreach my $bp (@bport) {
    $bp->{isglobal} = 1;
    if (grep(/$bp->{name}/, @hportnames)) {
        $bp->{isglobal} = 0;
    }
    else {
        $bp->{isglobal} = 1;
    }
}

=pod
print "port    dir    type    isglobal\n";
foreach (@bport) {
    print "$_->{name}    $_->{dir}    $_->{type}    $_->{isglobal}\n";
}

print "ifpga template: $ifpgafile\n";
=cut

outputIfpga($ifpgafile, \@bport);



=pod
    generate ifpga VHDL source.
    the source is based on a template $ifpgafile.
    additional port connections are inserted to the template
    according to the backend port information.
=cut
sub outputIfpga {
    my($infile, @bport, $infp, $line, $found);
    $infile = shift;
    $bpr = shift;
    @bport = @$bpr;

    open($infp,  $infile)   or die "Can't open $infile: $!\n";

    # insert ports to entity statement
    $found = 0;
    while ($line = <$infp>) {
        print $line;
        last if ($line =~ /port\s*\(/);
    }
    foreach (@bport) {
        if ($_->{isglobal}) {
            if (!$found) {
                $found = 1;
#                print "    -- connection to external ports\n";
            }

            printf("    %-23s: $_->{dir} $_->{type};\n", $_->{name});
        }
    }
    if ($found) {
        $found = 0;
        print "\n";
    }

    # insert ports to component statement
    while ($line = <$infp>) {
        print $line;
        last if ($line =~ /component backend/);
    }
    while ($line = <$infp>) {
        print $line;
        last if ($line =~ /port\s*\(/);
    }
    foreach (sort {$a->{name} cmp $b->{name}} @bport) {
        if ($_->{isglobal}) {
            if (!$found) {
                $found = 1;
#                print "      -- connection to external ports\n";
            }
            printf("      %-21s: $_->{dir} $_->{type};\n", $_->{name});
        }
    }
    if ($found) {
        $found = 0;
        print "\n";
    }
    foreach (sort {$a->{name} cmp $b->{name}} @bport) {
        if ($_->{name} =~ /backend_clk[0-5]/) {
            print "      $_->{name}         : $_->{dir} $_->{type};\n"
        }
    }

    # insert ports to backend instance
    while ($line = <$infp>) {
        print $line;
        last if ($line =~ /backend_instance\s*:\s*backend/);
    }
    while ($line = <$infp>) {
        print $line;
        last if ($line =~ /port\s*map\s*\(/);
    }
    foreach (sort {$a->{name} cmp $b->{name}} @bport) {
        if ($_->{isglobal}) {
            if (!$found) {
                $found = 1;
#                print "      -- connection to external ports\n";
            }
            printf("      %-12s => $_->{name},\n", $_->{name});
#            print "      $_->{name} => $_->{name};\n"
        }
    }
    if ($found) {
        $found = 0;
        print "\n";
    }
    foreach (sort {$a->{name} cmp $b->{name}} @bport) {
        if ($_->{name} =~ /backend_clk[0-5]/) {
            print "      $_->{name} => $_->{name},\n"
        }
    }

    while ($line = <$infp>) {
        print $line;
    }
}



=pod
    parse backend entity.
    returns an array of 'port's.
    each 'port' consists of 'name', 'dir', 'type', and 'isglobal'.
=cut
sub parseBackend {
    my($infile, $infp, $line, $found, @iostr, @port, @portstr);
    $infile = shift;
    open($infp,  $infile)   or die "Can't open $_[0]: $!\n";

# search for the beginning of entity declaration
    $found = 0;
    while ($line = <$infp>) {
        chomp($line);
        if ($line =~ /entity backend is/) {
            $found = 1;
        }
        last if ($found);
    }
    if (!$found) {
        die "Entiry backend not found.\n";
    }

# search for the beginning of port list
    $found = 0;
    while ($line = <$infp>) {
        chomp($line);
        last if ($line =~ /end backend/);

        if ($line =~ /port/) {
            $found = 1;
        }
        last if ($found);
    }
    if (!$found) {
        die "Entiry backend has no port.\n";
    }

# make a list of ports
    $found = 0;
    while ($line = <$infp>) {
        chomp($line);
        last if ($line =~ /end backend/);
        next if ($line =~ /^\s*--/);
        next if ($line =~ /^[^:]*$/);
        $line =~ s/(^.*);.*$/$1/g;
        $line =~ s/(^.*)\s*--.*$/$1/g;
        $line =~ s/(^.*\))\s*\).*$/$1/g;
        push @portstr, $line;
    }
    close($infp);

    my($name, $dir, $type);
    my $i = 0;
    foreach (@portstr) {
        $_ =~ /^\s*(\S*)\s*:\s*(\S*)\s+(.*)$/;
        $name = $1;
        $dir = $2;
        $type = $3;
        $port[$i++] = {
            name => $name,
            dir => $dir,
            type => $type,
            isglobal => 0,
        };
    }
    return @port;
}
