--
-- template: pg_conv_float_to_int
--
-- <DELAY=3>
--
-- convert a <$w1>-bit floating-point to a <$w2>-bit int.
--
-- src: float
-- 1     <$w1-$m1-1>            <$m1>
-- sign  exponent     mantissa
--       (with offset)
--
-- dst: <$w2>-bit int
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

-- w1=17  m1=9
-- w2=57

entity <$mname> is
  port (src    : in  std_logic_vector(<$w1-1> downto 0);
        dst    : out std_logic_vector(<$w2-1> downto 0);
        clk    : in  std_logic);
end <$mname>;

architecture rtl of <$mname> is

  component lpm_add_sub
    generic (LPM_WIDTH: integer;
             LPM_DIRECTION: string);
    port (dataa,datab: in std_logic_vector(LPM_WIDTH-1 downto 0);
          result: out std_logic_vector(LPM_WIDTH-1 downto 0));
  end component;

  component shift_fptofix_<$mid>
    port( indata : in std_logic_vector(<$m1> downto 0);
          control : in std_logic_vector(<$w1-$m1-2> downto 0);
          outdata : out std_logic_vector(<$w2-2> downto 0));
  end component;

  signal sign  : std_logic_vector(2 downto 0);
  signal nz0   : std_logic;
  signal e0, e0a, e1    : std_logic_vector(<$w1-$m1-2> downto 0);
  signal m0    : std_logic_vector(<$m1-1> downto 0);
  signal m1    : std_logic_vector(<$m1> downto 0);
  signal m1a   : std_logic_vector(<$w2-2> downto 0);
  signal m2    : std_logic_vector(<$w2-1> downto 0);

begin

  sign(0) <= src(<$w1-1>);            -- sign
  e0      <= src(<$w1-2> downto <$m1>);   -- exponent
  m0      <= src(<$m1-1> downto 0);    -- mantissa

  with src(<$w1-2> downto 0) select
    nz0 <=
    '0' when  conv_std_logic_vector(0, <$w1-1>),
    '1' when  others;

  process(clk) begin
    if(clk'event and clk='1') then
      sign(2 downto 1) <= sign(1 downto 0);
    end if;
  end process;

  -- remove offset.
  u1: lpm_add_sub generic map (LPM_WIDTH     => <$w1 - $m1 - 1>,
                               LPM_DIRECTION => "SUB")
                  port map(result => e0a,
                           dataa  => e0,
                           datab  => '1' & conv_std_logic_vector(0, <$w1 - $m1 - 2>));

  process(clk)
  begin
    if(clk'event and clk='1') then
      e1 <= e0a;
      m1 <= nz0 & m0;                   -- add hidden MSB
    end if;
  end process;

  u0: shift_fptofix_<$mid>
      port map (indata  => m1,
                control => e1,
                outdata => m1a);

  process(clk)
  begin
    if (clk'event and clk='1') then
      m2 <= '0' & m1a;                        -- m2 == |dst|
    end if;
  end process;

  process(clk)
  begin
    if (clk'event and clk='1') then
      if (sign(2) = '0') then
        dst(<$w2-1> downto 0) <= m2;
      else
        dst(<$w2-1> downto 0) <= (not m2) + conv_std_logic_vector(1, <$w2-1>);
      end if;
    end if;
  end process;

end rtl;

library ieee;
use ieee.std_logic_arith.all;
use ieee.std_logic_1164.all;

entity shift_fptofix_<$mid> is
  port( indata : in std_logic_vector(<$m1> downto 0);
        control : in std_logic_vector(<$w1-$m1-2> downto 0);
        outdata : out std_logic_vector(<$w2-2> downto 0));
end shift_fptofix_<$mid>;

architecture rtl of shift_fptofix_<$mid> is

<PG2>
  {
    # signal declaration part

    my $outtext = "";
    my $cwidth = width_of($w2);
    for my $i (0 .. $cwidth - 1) {
        $outtext .= "  signal c$i : std_logic_vector($cwidth downto 0);\n";
    }
    for my $i (0 .. $cwidth) {
        $outtext .= "  signal o$i, o" . $i. "d : std_logic_vector(" . ((2**$i) + $m1 - 1) . " downto 0);\n";
    }
    return $outtext;
}
</PG2>


begin

--
-- indata : s eeeeeee 1mmmmmmmmm
-- o6:                                               outdata:
--          s 00000000..1mmmmmmmmm,  if e=0          s 000000000....1
--          s 0000000..1mmmmmmmmm0,  if e=1          s 00000000....1m
--          ......................
--          s 0000..1mmmmmmmmm0..0,  if e=$m1        s 00..1mmmmmmmmm
--          s 0000..1mmmmmmmm00..0,  if e=$m1+1      s 0..1mmmmmmmmm0
--          ......................
--          s 1mmmmmmmmm..00000000,  if e=$w2-2      s 1mmmmmmmmm..00

  o0d <= indata;
<PG2>
  {
    my $outtext = "";
    my $cwidth = width_of($w2);

    $outtext .= sprintf qq{  c0 <= control(%d downto 0);\n\n}, $cwidth;
    for my $i (0 .. $cwidth - 1) {
        $outtext .= "  with c$i($i) select\n";
        $outtext .= sprintf qq{    o%d <= o%dd & conv_std_logic_vector(0, %d)      when '1',\n}, $i+1, $i, 2**$i;
        $outtext .= sprintf qq{                conv_std_logic_vector(0, %d) & o%dd when others;\n\n}, 2**$i, $i;
    }
    my $i = $cwidth;
    $outtext .= sprintf qq{  with c%d(%d) select\n}, $i-1, $i;
    $outtext .= sprintf qq{    outdata <= o%d(%d downto %d)            when '0',\n}, $i, ($w2 - 1) + $m1 - 1, $m1;
    $outtext .= sprintf qq{               conv_std_logic_vector(0, %d) when others;\n\n}, $w2-1;

    for my $i (1 .. $cwidth - 1) {
        $outtext .= sprintf qq{    o%dd <= o%d;\n}, $i, $i;
        $outtext .= sprintf qq{    c%d  <= c%d;\n\n}, $i, $i-1;
    }

    return $outtext;
}

</PG2>

end rtl;
