--
-- template: pg_conv_float_to_log
--
-- <DELAY=2>
--
-- convert a <$w1>-bit float to a <$w2>-bit log.
-- using a table <$tablefilename>.
--
-- width of exponent of dst should be larger than or equal to that of
-- src. otherwise the exponent may overflow.
--
--
-- float:
-- 1     <$e1>        <$m1>
-- sign  exponent     mantissa
--
-- log:
-- 1     <$e2>        <$m2>
-- sign  exponent     mantissa
--
--    tablefile: <$tablefilename>
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity <$mname> is
  port (src    : in  std_logic_vector(<$w1-1> downto 0);
        dst    : out std_logic_vector(<$w2-1> downto 0);
        clk    : in  std_logic);
end <$mname>;

architecture rtl of <$mname> is

  component table_unreg
    generic (IN_WIDTH   : integer ;
             OUT_WIDTH  : integer ;
             TABLE_FILE : string);
    port (indata : in  std_logic_vector(IN_WIDTH-1 downto 0);
         outdata : out std_logic_vector(OUT_WIDTH-1 downto 0);
         clk     : in  std_logic);
  end component;

  component lpm_add_sub
    generic (LPM_WIDTH: integer;
             LPM_DIRECTION: string);
    port (dataa,datab: in std_logic_vector(LPM_WIDTH-1 downto 0);
          result: out std_logic_vector(LPM_WIDTH-1 downto 0));
  end component;

  signal sign0, sign1, sign2 : std_logic;
  signal nz1, nz2     : std_logic;
  signal e0, e1       : std_logic_vector(<$e1-1> downto 0);
  signal e2           : std_logic_vector(<$e2-1> downto 0);
  signal m0           : std_logic_vector(<$m1-1> downto 0);
  signal m1, m2       : std_logic_vector(<$m2> downto 0);
  signal dst2         : std_logic_vector(<$w2-1> downto 0);
  
begin

  sign0 <= src(<$w1-1>);               -- sign
  e0    <= src(<$w1-2> downto <$m1>);      -- exponent
  m0    <= src(<$m1-1> downto 0);       -- mantissa

  process(clk)                    -- non zero
  begin
    if(clk'event and clk='1') then
      if (src(<$w1-2> downto 0) = conv_std_logic_vector(0, <$w1-1>)) then
        nz1 <= '0';
      else
        nz1 <= '1';
      end if;
      nz2 <= nz1;
    end if;
  end process;

  process(clk)
  begin
    if(clk'event and clk='1') then
      sign1 <= sign0;
      sign2 <= sign1;
    end if;
  end process;

  -- an extra 1 bit at MSB of table output indicates overflow
  -- which may occur when input is very close to 2.0.
  -- the overflow is adjusted by adding 1 to the exponent.
  u1: table_unreg
    generic map (IN_WIDTH   => <$m1>,
                 OUT_WIDTH  => <$m2+1>,
                 TABLE_FILE => <$tablefilename>)
    port map (indata  => m0,
              outdata => m1,  -- 1-clk delay
              clk     => clk);
  
  -- exponent
  process(clk)
  begin
    if(clk'event and clk='1') then
      e1 <= e0 - ('1' & conv_std_logic_vector(0, <$e1-1>));  -- remove offset.
      e2 <= (conv_std_logic_vector(0, <$e2-$e1>) & e1) +
            ('1' & conv_std_logic_vector(0, <$e2-2>) & m1(<$m2>)); -- add offset and overflow.
    end if;
  end process;

  process(clk)
  begin
    if(clk'event and clk='1') then
      m2 <= m1;
    end if;
  end process;

  dst2(<$w2-1>)              <= sign2;
  dst2(<$w2-2> downto <$m2>) <= conv_std_logic_vector(0, <$e2-$e1>) & e2;
  dst2(<$m2-1> downto 0)     <= m2(<$m2-1> downto 0);

  with nz2 select
    dst(<$w2-1> downto 0) <=
      dst2                            when '1',
      conv_std_logic_vector(0, <$w2>) when others;

end rtl;
