--
-- template: pg_conv_int_to_float
--
-- <DELAY=3>
--
-- convert a <$w1>-bit int to a <$w2>-bit floating-point.
--
-- src: <$w1>-bit int
--
-- dst: float:
-- 1     <$w2-$m2-1>            <$m2>
-- sign  exponent     mantissa
--
-- example:
--------------------------------------------------------------------------------------------------
-- int                          float
--                              s exp   man
--------------------------------------------------------------------------------------------------
-- 0000_0000_0000_0000          0_00000_000000000     zero -> exp=zero, man=0.
-- 0000_0000_0000_0001          0_10000_000000000     one  -> exp=bias+0, man=0.
-- 0000_0111_1111_1111          0_11010_111111111     large number -> normalized & rounded.
-- 1111_1111_1111_1111          1_10000_000000000     minus one -> sign=1, exp=bias+0, man=0.
-- 1111_1111_1111_1000          1_10011_000000000     
--------------------------------------------------------------------------------------------------
--
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity <$mname> is
  port (src    : in  std_logic_vector(<$w1-1> downto 0);
        dst    : out std_logic_vector(<$w2-1> downto 0);
        clk    : in  std_logic);
end <$mname>;

architecture rtl of <$mname> is

  component penc_<$mid>_<$w1-1>
    port( a : in std_logic_vector(<$w1-2> downto 0);
          c : out std_logic_vector(<width_of($w1-1) - 1> downto 0));
  end component;

  component lpm_clshift
    generic (LPM_WIDTH : POSITIVE;
             LPM_WIDTHDIST : POSITIVE);
    port(data : in std_logic_vector(LPM_WIDTH-1 downto 0);
         distance : in std_logic_vector(LPM_WIDTHDIST-1 downto 0);
         direction : in std_logic;
         result : out std_logic_vector(LPM_WIDTH-1 downto 0));
  end component;

  component lpm_add_sub
    generic (LPM_WIDTH: integer;
             LPM_DIRECTION: string);
    port (dataa,datab: in std_logic_vector(LPM_WIDTH-1 downto 0);
          result: out std_logic_vector(LPM_WIDTH-1 downto 0));
  end component;

  component round_ubf1
    generic (IN_WIDTH   : integer;
             OUT_WIDTH  : integer
    );
    port (indata  : in  std_logic_vector(IN_WIDTH-1 downto 0);
          outdata : out std_logic_vector(OUT_WIDTH-1 downto 0)
    );
  end component;

  signal fminus0, fminus0a, fplus0 : std_logic_vector(<$w1-2> downto 0);
  signal f0, f1, f2                : std_logic_vector(<$w1-2> downto 0);
  signal s0, s0a, s1, s2     : std_logic;
  signal zero0, zero1, zero2 : std_logic;
  signal s0b    : std_logic_vector(1 downto 0);
  signal e1, e2 : std_logic_vector(<width_of($w1-1) - 1> downto 0);
  signal e2a    : std_logic_vector(<$w2 - $m2 - 2> downto 0);
  signal n1, n2 : std_logic_vector(<width_of($w1-1) - 1> downto 0);
  signal m1, m2 : std_logic_vector(<$w1 - 2> downto 0);
  signal m2a    : std_logic_vector(<$m2 - 1> downto 0);

begin

  -- stage 0
  s0      <=      src(<$w1-1>);
  fplus0  <=      src(<$w1-2> downto 0);
  fminus0 <=  not src(<$w1-2> downto 0);
  u1: lpm_add_sub
    generic map (LPM_WIDTH=><$w1-1>,LPM_DIRECTION=>"ADD")
    port map(result => fminus0a,
             dataa  => fminus0,
             datab  => conv_std_logic_vector(1, <$w1-1>));

  s0a <= '0' when src(<$w1-2> downto 0) = conv_std_logic_vector(0, <$w1-1>) else '1';
  s0b <= s0a & s0;
  
  with s0b select
    f0 <=
    fminus0a when "11",                 -- negative number
    fminus0 when "01",                  -- Nan (i.e. "10000..0") should be treated as "1111...11"
    fplus0 when others;                 -- positive number

  process(clk)
  begin
    if(clk'event and clk='1') then
      f1 <= f0;
      s1 <= s0;
    end if;
  end process;

  -- stage 1
  -- f1: |src|
  u2: penc_<$mid>_<$w1-1> port map (a=>f1, c=>e1);

  process(clk)
  begin
    if(clk'event and clk='1') then
      f2 <= f1(<$w1-2> downto 0);
      e2 <= e1;
      s2 <= s1;
    end if;
  end process;
  

  -- this subtraction is not necessary if u2:penc is modified
  -- so that it returns width_of($w1-1)-e1.
  -- see negative_penc in pg_add_float
  u3: lpm_add_sub
    generic map (LPM_WIDTH => <width_of($w1-1)>,
                 LPM_DIRECTION => "SUB"
    )
    port map(result => n2,
             dataa  => "<sprintf("%0" . width_of($w1-1) . "b", $w1-1)>",
             datab  => e2
    );

  u4: lpm_clshift
    generic map (LPM_WIDTH => <$w1-1>,
                 LPM_WIDTHDIST => <width_of($w1-1)>)
    port map (data=>f2,
              distance=>n2,
              result=>m2,
              direction=>'0');

  u5: round_ubf1
    generic map (IN_WIDTH  => <$w1 - 1>,
                 OUT_WIDTH => <$m2>
    )
    port map (indata  => m2,
              outdata => m2a
    );

  -- add bias.
  u6: lpm_add_sub
    generic map (LPM_WIDTH => <$w2 - $m2 - 1>,
                 LPM_DIRECTION => "ADD"
    )
    port map(result => e2a,
             dataa  => '1' & conv_std_logic_vector(0, <$w2 - $m2 - 2>),
             datab  => conv_std_logic_vector(0, <$w2 - $m2 - 1 - width_of($w1-1)>) & e2
    );
  
  zero0 <= '1' when src = conv_std_logic_vector(0, src'length) else '0';
  
  process(clk)
  begin
    if(clk'event and clk='1') then
      zero1 <= zero0;
      zero2 <= zero1;
    end if;
  end process;

  process(clk)
  begin
    if(clk'event and clk='1') then
      if (zero2 = '1') then
        dst <= (others => '0');
      else
        dst(<$w2 - 1>)              <= s2;
        dst(<$w2 - 2> downto <$m2>) <= e2a;
        dst(<$m2 - 1> downto 0)     <= m2a;
      end if;
    end if;
  end process;

  -- stage 3
  -- dst: number converted to floating-point format

end rtl;


library ieee;
use ieee.std_logic_1164.all;

entity penc_<$mid>_<$w1-1> is
port( a : in std_logic_vector(<$w1-2> downto 0);
      c : out std_logic_vector(<width_of($w1-1) - 1> downto 0));
end penc_<$mid>_<$w1-1>;

architecture rtl of penc_<$mid>_<$w1-1> is
begin

  process(a)
  begin
    <PG2>
    {
        my $inwidth = $w1-1;
        my $cwidth = width_of($inwidth);
        my $outtext .= q{if (a(} . ($inwidth-1) . qq{)='1') then\n};
        my $fmt = qq{      c <= "%0} . $cwidth . qq{b";\n};
        for my $i (reverse 1..$inwidth-2) {
            $outtext .= sprintf($fmt, $i+1);
            $outtext .= "    elsif (a($i) = '1') then\n";
        }
        $outtext .= sprintf($fmt, 1);
        $outtext .= "    else\n";
        $outtext .= sprintf($fmt, 0);
        $outtext .= "    end if;\n";
        return $outtext;
    }
    </PG2>
  end process;
end rtl;
