library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_arith.all;
library lpm;
use lpm.lpm_components.all;

entity pg_ctl is
    generic (
        JDATA_WIDTH  : integer;
        JDATA_DEPTH  : integer;
        JMEM_BLOCKS  : integer
        );
    port (
        -- PCI-X side
        hib_we       : in std_logic;
        hib_data     : in std_logic_vector(63 downto 0);
        backend_we   : out std_logic;
        backend_data : out std_logic_vector(63 downto 0);
        rst          : in std_logic;
        clk          : in std_logic;

        -- pg_pipe side
        p_jdata      : out std_logic_vector(JDATA_WIDTH-1 downto 0);
        p_run        : out std_logic;
        p_runret     : in std_logic;
        p_bcast      : out std_logic;
        p_we         : out std_logic;
        p_adri       : out std_logic_vector(11+4 downto 0);
        p_datai      : out std_logic_vector(63 downto 0);
        p_adro       : out std_logic_vector(11+4 downto 0);
        p_datao      : in std_logic_vector(63 downto 0)
    );
end pg_ctl;

architecture hierarchy of pg_ctl is

    component ipw
        generic (
            JDATA_DEPTH : integer;
            JMEM_BLOCKS : integer
        );
        port (
            hib_we   : in std_logic;
            hib_data : in std_logic_vector(63 downto 0);
            m_we     : out std_logic;
            m_cs     : out std_logic_vector(3 downto 0);
            m_adr    : out std_logic_vector(JDATA_DEPTH-1 downto 0);
            m_data   : out std_logic_vector(63 downto 0);
            p_we     : out std_logic;
            p_adri   : out std_logic_vector(11+4 downto 0);
            p_datai  : out std_logic_vector(63 downto 0);
            i_ipwon  : out std_logic;
            clk      : in std_logic;
            rst      : in std_logic
        );
    end component;

    component calc
        generic (
            JDATA_DEPTH: integer
        );
        port (
            hib_we   : in std_logic;
            hib_data : in std_logic_vector(63 downto 0);
            c_adr    : out std_logic_vector(JDATA_DEPTH-1 downto 0);
            p_run    : out std_logic;
            i_ipwon  : in std_logic;
            rst      : in std_logic;
            clk      : in std_logic
        );
    end component;

    component fo
        port (
            hib_we       : in std_logic;
            hib_data     : in std_logic_vector(63 downto 0);
            backend_we   : out std_logic;
            backend_data : out std_logic_vector(63 downto 0);
            p_adro       : out std_logic_vector(11+4 downto 0);
            p_datao      : in std_logic_vector(63 downto 0);
            p_runret     : in std_logic;
            i_ipwon      : in std_logic;
            rst          : in std_logic;
            clk          : in std_logic
        );
    end component;

    component pg_dpram
        port (
            data      : in std_logic_vector (63 downto 0);
            wren      : in std_logic := '1';
            wraddress : in std_logic_vector (JDATA_DEPTH-1 downto 0);
            rdaddress : in std_logic_vector (JDATA_DEPTH-1 downto 0);
            clock     : in std_logic;
            q         : out std_logic_vector (63 downto 0)
        );
    end component;

    signal m_wes       : std_logic_vector(JMEM_BLOCKS-1 downto 0);
    signal m_we        : std_logic;
    signal m_cs        : std_logic_vector(3 downto 0);
    signal m_adr       : std_logic_vector(JDATA_DEPTH-1 downto 0);
    signal m_data      : std_logic_vector(63 downto 0);
    signal c_adr       : std_logic_vector(JDATA_DEPTH-1 downto 0);
    signal i_ipwon     : std_logic;
    signal hib_we1     : std_logic;
    signal hib_data1   : std_logic_vector(63 downto 0);
    signal ipw_adri    : std_logic_vector(11+4 downto 0);
    signal ipw_datai   : std_logic_vector(63 downto 0);
    signal coeff_adri  : std_logic_vector(7 downto 0);  -- 8-bit address
    signal coeff_datai : std_logic_vector(31-coeff_adri'length downto 0);  -- 24-bit data
    signal bcast       : std_logic;

begin

    uipw : ipw generic map (
                   JDATA_DEPTH => JDATA_DEPTH,
                   JMEM_BLOCKS => JMEM_BLOCKS
               )
               port map (
                   hib_we   => hib_we,
                   hib_data => hib_data,
                   m_we     => m_we,
                   m_cs     => m_cs,
                   m_adr    => m_adr,
                   m_data   => m_data,
                   p_we     => p_we,
                   p_adri   => ipw_adri,
                   p_datai  => ipw_datai,
                   i_ipwon  => i_ipwon,
                   clk      => clk,
                   rst      => rst
               );

    ucalc: calc generic map (
                    JDATA_DEPTH => JDATA_DEPTH
                )
                port map (
                    hib_we   => hib_we,
                    hib_data => hib_data,
                    c_adr    => c_adr,
                    p_run    => p_run,
                    i_ipwon  => i_ipwon,
                    rst      => rst,
                    clk      => clk
                );

    ufo: fo port map (
                hib_we       => hib_we,
                hib_data     => hib_data,
                backend_we   => backend_we,
                backend_data => backend_data,
                p_adro       => p_adro,
                p_datao      => p_datao,
                p_runret     => p_runret,
                i_ipwon      => i_ipwon,
                rst          => rst,
                clk          => clk
            );
    for_we: for i in 0 to JMEM_BLOCKS-1 generate
        process (m_we, m_cs)
        begin
            if (m_we = '1') then
                if(m_cs = conv_std_logic_vector(i, m_cs'length)) then
                    m_wes(i) <= '1';
                else
                    m_wes(i) <= '0';
                end if;
            else
                m_wes(i) <= '0';
            end if;
        end process;
    end generate for_we;

    for_dpram: for i in 0 to JMEM_BLOCKS-1 generate
        udpram : pg_dpram port map (
                           data          => m_data,
                           wren          => m_wes(i),
                           wraddress     => m_adr,
                           rdaddress     => c_adr,
                           clock         => clk,
                           q             => p_jdata(64*i+63 downto 64*i+0)
                       );
    end generate for_dpram;

    process (clk)
    begin
        if (clk'event and clk='1') then
            hib_we1 <= hib_we;
            hib_data1 <= hib_data;
        end if;
    end process;

    p_bcast <= bcast;
    process (clk, rst)
    begin
        if (rst = '0') then
            bcast <= '0';
        elsif (clk'event and clk='1') then
            if (hib_we1 ='1' and hib_data1(31 downto 28)="1101" and i_ipwon='0') then
                bcast <= '1';
                coeff_adri  <= hib_data1(63 downto 63 - coeff_adri'length + 1); -- 8-bit address
                coeff_datai <= hib_data1(63 - coeff_adri'length downto 32);     -- 24-bit data
            else
                bcast <= '0';
            end if;
        end if;
    end process;

    p_adri <= conv_std_logic_vector(0, p_adri'length - coeff_adri'length) & coeff_adri when bcast = '1'
              else ipw_adri;

    p_datai <= conv_std_logic_vector(0, p_datai'length - coeff_datai'length) & coeff_datai when bcast = '1'
               else ipw_datai;

end hierarchy;

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;

entity ipw is
    generic (
        JDATA_DEPTH : integer;
        JMEM_BLOCKS : integer
    );
    port (
        hib_we   : in std_logic;
        hib_data : in std_logic_vector(63 downto 0);
        m_we     : out std_logic;
        m_cs     : out std_logic_vector(3 downto 0);
        m_adr    : out std_logic_vector(JDATA_DEPTH-1 downto 0);
        m_data   : out std_logic_vector(63 downto 0);
        p_we     : out std_logic;
        p_adri   : out std_logic_vector(11+4 downto 0);
        p_datai  : out std_logic_vector(63 downto 0);
        i_ipwon  : out std_logic;
        clk      : in std_logic;
        rst      : in std_logic
    );
end ipw;

architecture rtl of ipw is

    signal hib_we1 : std_logic;
    signal hib_data1 : std_logic_vector(63 downto 0);
    signal iword_dc : std_logic_vector(15 downto 0);
    signal iword_uc : std_logic_vector(15 downto 0);
    signal ipwon,iflag,jflag: std_logic;
    signal ni,nd : std_logic_vector(15 downto 0);
    signal ah_dc,ah_uc : std_logic_vector(7 downto 0);    -- higher 8-bit address for pipe select
    signal al_dc,al_uc : std_logic_vector(3+4 downto 0);  -- lower 8-bit address for a pipe internal space
    signal m_adr0 : std_logic_vector(JDATA_DEPTH-1 downto 0);
    signal m_cs0  : std_logic_vector(3 downto 0);

begin

--
-- hib_we      __~~~~~~~~_____
-- hib_data    --<      >-----
-- hib_we1       __~~~~~~~~_____
-- hib_data1     --<      >-----
-- word_dc      -----<><><>-----
--

    process (clk)
    begin
        if (clk'event and clk='1') then
            hib_we1 <= hib_we;
            hib_data1 <= hib_data;
        end if;
    end process;

    process (clk, rst)
    begin
        if (rst = '0') then
            ni <= (others => '0');
            nd <= (others => '0');
        elsif (clk'event and clk='1') then
            if(hib_we1 ='1' and hib_data1(31 downto 28)="1100" and ipwon='0') then
                ni <= hib_data1(47 downto 32);
                nd <= hib_data1(63 downto 48);
            end if;
        end if;
    end process;

    -- i word counter

    process (iword_dc)
    begin
        if(iword_dc /= conv_std_logic_vector(0, iword_dc'length)) then
            ipwon <= '1';
        else
            ipwon <= '0';
        end if;
    end process;

    i_ipwon <= ipwon;

    process (clk, rst)
    begin
        if (rst = '0') then
            iword_dc <= (others => '0');
        elsif (clk'event and clk='1') then
            if (hib_we1 = '1') then
                if (iword_dc = conv_std_logic_vector(0 , iword_dc'length)) then
                    if (hib_data1(31 downto 28)="0000") then
                        iword_dc <= hib_data1(47 downto 32);             -- num of IPs in 64-bit words
                        iword_uc <= hib_data1(63 downto 48);             -- not used
                        iflag <= '1';
                        jflag <= '0';
                    elsif (hib_data1(31 downto 28)="0100") then
                        iword_dc <= hib_data1(47 downto 32);             -- num of JPs in 64-bit words
                        iword_uc <= hib_data1(63 downto 48);             -- start addr in 64-bit words
                        iflag <= '0';
                        jflag <= '1';
                    end if;
                else
                    iword_dc <= iword_dc - conv_std_logic_vector(1, iword_dc'length);
                    iword_uc <= iword_uc + conv_std_logic_vector(1, iword_uc'length);
                end if;
            end if;
        end if;
    end process;

    -- jp

    process (clk, rst)
    begin
        if (rst = '0') then
            m_cs0  <= (others => '0');
            m_adr0 <= (others => '0');
        elsif (clk'event and clk='1') then
            if (hib_we1 = '1') then
                if (iword_dc = conv_std_logic_vector(0, iword_dc'length)) then
                    if (hib_data1(31 downto 28)="0100") then
                        m_cs0 <= conv_std_logic_vector(0, m_cs0'length); -- start from the 1st memory block
                        m_adr0 <= hib_data1(JDATA_DEPTH+48-1 downto 48); -- start addr in JDATA_WIDTH-bit words
                    end if;
                elsif (jflag = '1') then
                    m_cs  <= m_cs0;
                    m_adr <= m_adr0;
                    if (m_cs0 = conv_std_logic_vector(JMEM_BLOCKS-1, m_cs0'length)) then -- adress++
                        m_cs0 <= conv_std_logic_vector(0, m_cs0'length);
                        m_adr0 <= m_adr0 + conv_std_logic_vector(1, JDATA_DEPTH);
                    else
                        m_cs0 <= m_cs0 + conv_std_logic_vector(1, m_cs0'length);
                    end if;
                end if;
            end if;
        end if;
    end process;

    process (clk)
    begin
        if (rst = '0') then
            m_we <= '0';
        elsif (clk'event and clk='1') then
            if (ipwon = '1' and jflag = '1' and hib_we1 = '1') then
                m_we <= '1';
                m_data <= hib_data1;
            else
                m_we <= '0';
            end if;
        end if;
    end process;

    -- ip

    process (clk)
    begin
        if (clk'event and clk='1') then
            if (ipwon='1' and iflag = '1' and hib_we1 = '1') then
                p_we <= '1';
                p_datai <= hib_data1;
                p_adri <= ah_uc & al_uc;
            else
                p_we <= '0';
            end if;
        end if;
    end process;

    process (clk, rst)
    begin
        if (rst = '0') then
             ah_dc <= (others => '0');
        elsif (clk'event and clk='1') then
            if (hib_we1 = '1' and hib_data1(31 downto 28)="0000"
                and iword_dc = conv_std_logic_vector(0 ,16)) then
                ah_dc <= ni(ah_dc'high downto 0);
                ah_uc <= conv_std_logic_vector(0, ah_uc'length);
            elsif (al_dc  = conv_std_logic_vector(1, al_dc'length) and
                   ah_dc /= conv_std_logic_vector(0, ah_dc'length)) then
                ah_dc <= ah_dc - conv_std_logic_vector(1, ah_dc'length);
                ah_uc <= ah_uc + conv_std_logic_vector(1, ah_uc'length);
            end if;
        end if;
    end process;

    process (clk, rst)
    begin
        if (rst = '0') then
            al_dc <= (others => '0');
        elsif (clk'event and clk='1') then
            if (hib_we1 = '1' and hib_data1(31 downto 28)="0000"
                and iword_dc = conv_std_logic_vector(0 ,16)) then
                al_dc <= nd(al_dc'high downto 0);
                al_uc <= conv_std_logic_vector(0, al_uc'length);
            elsif (al_dc  = conv_std_logic_vector(1, al_dc'length) and
                   ah_dc /= conv_std_logic_vector(0, ah_dc'length)) then
                al_dc <= nd(al_dc'high downto 0);
                al_uc <= conv_std_logic_vector(0, al_uc'length);
            elsif (al_dc /= conv_std_logic_vector(0, al_dc'length)) then
                al_dc <= al_dc - conv_std_logic_vector(1, al_dc'length);
                al_uc <= al_uc + conv_std_logic_vector(1, al_uc'length);
            end if;
        end if;
    end process;

end rtl;

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;

entity calc is
    generic(JDATA_DEPTH: integer);
    port(
      hib_we: in std_logic;
      hib_data : in std_logic_vector(63 downto 0);
      c_adr : out std_logic_vector(JDATA_DEPTH-1 downto 0);
      p_run : out std_logic;
      i_ipwon : in std_logic;
      rst,clk : in std_logic);
end calc;

architecture rtl of calc is

signal hib_we1: std_logic;
signal hib_data1 : std_logic_vector(63 downto 0);
signal ncnt_dc : std_logic_vector(JDATA_DEPTH-1+1 downto 0);  -- needs JDATA_DEPTH+1 bit
signal ncnt_uc : std_logic_vector(JDATA_DEPTH-1 downto 0);
signal run0 : std_logic;

begin

--
-- hib_we      __~~_____
-- hib_data    --<> -----
-- hib_we1       __~~_____
-- hib_data1     --<>-----
-- ncnt_dc      -----<3<2<1<0---
-- ncnt_uc      -----<0<1<2<3---
-- c_adr        -----<0<1<2-----
-- p_jdata      ---------<0<1<2-----
-- run          _______~~~~~~__
-- p_run        _______~~~~~~__


  process (clk)
  begin
      if (clk'event and clk='1') then
          hib_we1 <= hib_we;
          hib_data1 <= hib_data;
      end if;
  end process;

  process (clk, rst)
  begin
      if (rst = '0') then
          ncnt_dc <= (others => '0');
      elsif (clk'event and clk='1') then
          if (hib_we1 ='1' and hib_data1(31 downto 28)="1000" and i_ipwon='0') then
              ncnt_dc <= hib_data1(JDATA_DEPTH+32-1+1 downto 32);
              ncnt_uc <= (others => '0');
          elsif (ncnt_dc /= conv_std_logic_vector(0, ncnt_dc'length)) then
              ncnt_dc <= ncnt_dc - conv_std_logic_vector(1, ncnt_dc'length);
              ncnt_uc <= ncnt_uc + conv_std_logic_vector(1, ncnt_uc'length);
          end if;
      end if;
  end process;

  c_adr <= ncnt_uc(JDATA_DEPTH-1 downto 0);

  process (clk)
  begin
      if (clk'event and clk='1') then
          if (ncnt_dc /= conv_std_logic_vector(0, ncnt_dc'length)) then
              run0 <= '1';
          else
              run0 <= '0';
          end if;
      end if;
  end process;

  p_run <= run0;

end rtl;

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;

entity fo is
    port(
      hib_we: in std_logic;
      hib_data : in std_logic_vector(63 downto 0);
      backend_we: out std_logic;
      backend_data : out std_logic_vector(63 downto 0);
      p_adro : out std_logic_vector(11+4 downto 0);
      p_datao : in std_logic_vector(63 downto 0);
      p_runret : in std_logic;
      i_ipwon: in std_logic;
      rst,clk : in std_logic );
end fo;

architecture rtl of fo is

signal hib_we1: std_logic;
signal hib_data1 : std_logic_vector(63 downto 0);
signal ni,nd : std_logic_vector(15 downto 0);
signal ah_dc,ah_uc : std_logic_vector(8 downto 0);
signal al_dc,al_uc : std_logic_vector(4+4 downto 0);
signal runret1, we : std_logic;

begin

-- p_runret   ~~____________
-- runret1    ~~~~__________
-- ah_dc      ----<2 ><1 ><0
-- ah_uc      ----<0 ><1 ><2
-- al_dc      ----<2<1<2<1<2<1<0
-- al_uc      ----<0<1<0<1<0<1<2
-- p_adro     ----<      >------
-- p_datao    ------<      >----
-- backend    --------<      >----
--   _data
-- backend_we ________~~~~~~~~____


  process (clk)
  begin
      if(clk'event and clk='1') then
          hib_we1 <= hib_we;
          hib_data1 <= hib_data;
      end if;
  end process;

  process (clk, rst)
  begin
      if (rst = '0') then
          ni <= (others => '0');
          nd <= (others => '0');
      elsif (clk'event and clk='1') then
          if(hib_we1 ='1' and hib_data1(31 downto 28)="1110" and i_ipwon='0') then
              ni <= hib_data1(47 downto 32);
              nd <= hib_data1(63 downto 48);
          end if;
      end if;
  end process;

  process (clk)
  begin
      if(clk'event and clk='1') then
          runret1 <= p_runret;
      end if;
  end process;

  process (clk, rst)
  begin
      if (rst = '0') then
          ah_dc <= (others => '0');
      elsif (clk'event and clk='1') then
          if (p_runret = '0' and runret1 = '1') then
              ah_dc <= ni(ah_dc'high downto 0);
              ah_uc <= conv_std_logic_vector(0, ah_uc'length);
          elsif (al_dc  = conv_std_logic_vector(1, al_dc'length) and
                 ah_dc /= conv_std_logic_vector(0, ah_dc'length)) then
              ah_dc <= ah_dc - conv_std_logic_vector(1, ah_dc'length);
              ah_uc <= ah_uc + conv_std_logic_vector(1, ah_uc'length);
          end if;
      end if;
  end process;

  process (clk, rst)
  begin
      if (rst = '0') then
          al_dc <= (others => '0');
      elsif (clk'event and clk='1') then
          if(p_runret = '0' and runret1 = '1') then
              al_dc <= nd(al_dc'high downto 0);
              al_uc <= (others => '0');
          elsif(al_dc  = conv_std_logic_vector(1, al_dc'length) and
                ah_dc /= conv_std_logic_vector(0, ah_dc'length)) then
              al_dc <= nd(al_dc'high downto 0);
              al_uc <= (others => '0');
          elsif(al_dc /= conv_std_logic_vector(0, al_dc'length)) then
              al_dc <= al_dc - conv_std_logic_vector(1, al_dc'length);
              al_uc <= al_uc + conv_std_logic_vector(1, al_uc'length);
          end if;
      end if;
  end process;

  p_adro <= ah_uc(7 downto 0) & al_uc(3+4 downto 0);

  process (clk)
  begin
      if(clk'event and clk='1') then
          if(ah_dc /= conv_std_logic_vector(0, ah_dc'length)) then
              we <= '1';
          else
              we <= '0';
          end if;
      end if;
  end process;

  process (clk)
  begin
      if (clk'event and clk='1') then
          backend_we <= we;
          backend_data <= p_datao;
      end if;
  end process;

end rtl;
