--
-- template: pg_cutoff_log_log_log
--
-- <DELAY=5>
--
-- cutoff function for P^3M
-- inputs should be positive and non-zero.
--
-- r, eta, ginv
-- log:
-- 1    1         <$w1-$m1-2>            <$m1>
-- sign non-zero  exponent     mantissa
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity <$mname> is
  port (r    : in  std_logic_vector(<$w1-1> downto 0);
        eta  : in  std_logic_vector(<$w2-1> downto 0);
        ginv : out std_logic_vector(<$w3-1> downto 0);
        clk  : in  std_logic
  );
end <$mname>;

architecture rtl of <$mname> is

  component lpm_add_sub
    generic (LPM_WIDTH     : integer;
             LPM_DIRECTION : string
    );
    port (dataa  : in  std_logic_vector(LPM_WIDTH-1 downto 0);
          datab  : in  std_logic_vector(LPM_WIDTH-1 downto 0);
          result : out std_logic_vector(LPM_WIDTH-1 downto 0)
    );
  end component;

  component lpm_clshift
    generic (LPM_WIDTH     : POSITIVE;
             LPM_WIDTHDIST : POSITIVE
    );
    port (data      : in  std_logic_vector(LPM_WIDTH-1 downto 0);
          distance  : in  std_logic_vector(LPM_WIDTHDIST-1 downto 0);
          direction : in  std_logic;
          result    : out std_logic_vector(LPM_WIDTH-1 downto 0)
    );
  end component;

  component conv_log_to_float_17_8_17_9
    port (src : in std_logic_vector(16 downto 0);
          dst : out std_logic_vector(16 downto 0);
          clk : in std_logic
    );
  end component;

  component table_unreg
    generic (IN_WIDTH   : integer;
             OUT_WIDTH  : integer;
             TABLE_FILE : string
             );
    port (indata  : in  std_logic_vector(IN_WIDTH-1 downto 0);
          outdata : out std_logic_vector(OUT_WIDTH-1 downto 0);
          clk     : in  std_logic
          );
  end component;

  signal xe, xeoff0     : std_logic_vector( 6 downto 0);
  signal xoff0          : std_logic_vector(14 downto 0);
  signal xylog0, xylog1 : std_logic_vector(14 downto 0);
  signal xyfp3          : std_logic_vector(16 downto 0);
  signal m3, m3a        : std_logic_vector(17 downto 0);
  signal e3             : std_logic_vector( 6 downto 0);
  signal nz1, nz2 ,nz3  : std_logic;
  signal ovflw2, ovflw3 : std_logic;
  signal entry4         : std_logic_vector( 8 downto 0);
  signal ginv5          : std_logic_vector(11 downto 0);
  signal gnz4, gnz5     : std_logic;

begin

  xe <= r(14 downto 8);                 -- exponent
    
  -- add 8-bit offset to r
  -- pipeline delay: 0
  u0: lpm_add_sub
    generic map (LPM_WIDTH     => 7,
                 LPM_DIRECTION => "ADD"
    )
    port map (result => xeoff0,
              dataa  => xe,
              datab  => conv_std_logic_vector(8,7)
    );
  xoff0 <= xeoff0 & r(7 downto 0);

  -- 2^8 * r / eta in log fmt
  -- pipeline delay: 1
  u1: lpm_add_sub
    generic map (LPM_WIDTH     => 15,
                 LPM_DIRECTION => "SUB"
    )
    port map (dataa  => xoff0,
              datab  => eta(14 downto 0),
              result => xylog0
    );
  process (clk)
  begin
    if(clk'event and clk='1') then
      xylog1 <= xylog0;
      end if;
  end process;

  process (clk)
  begin
    if(clk'event and clk='1') then
      if (xylog1(13 downto 8) >= "001001") then
        ovflw2 <= '1';
      else
        ovflw2 <= '0';
      end if;
    end if;
  end process;

  with xylog1(14) select 
    nz1 <=
    '1' when '0',
    '0' when others;  --  x/y is smaller than 2^-8

  process (clk)
  begin
    if(clk'event and clk='1') then
      nz2 <= nz1;
      nz3 <= nz2;
    end if;
  end process;
  
  process (clk)
  begin
    if(clk'event and clk='1') then
      ovflw3 <= ovflw2;
    end if;
  end process;
  
  -- entry in log fmt -> fp fmt
  -- pipeline delay: 2
  u2: conv_log_to_float_17_8_17_9
    port map (src => "01" & xylog1,     -- add sign bit & non-zero bit
              dst => xyfp3,
              clk => clk
    );

  -- entry in fp fmt -> fix fmt
  -- pipeline delay: 1
  e3 <= xyfp3(15 downto 9);
  m3 <= conv_std_logic_vector(0,8) & '1' & xyfp3(8 downto 0);  -- add hidden MSB

  u3: lpm_clshift
    generic map (LPM_WIDTH     => 18,
                 LPM_WIDTHDIST => 4
    )
    port map (data      => m3,
              distance  => e3(3 downto 0),
              result    => m3a,
              direction => '0'
    );
  process(clk)
  begin
    if(clk'event and clk='1') then
      if (nz3 = '0') then
        entry4 <= "000000000";
      elsif (ovflw3 = '1') then
        entry4 <= "111111111";
      else
        entry4 <= m3a(17 downto 9);
      end if;
    end if;
  end process;

  -- obtain 1/g by table lookup
  -- table entry    : entry4
  -- table output   : ginv1
  -- pipeline delay : 1
  u4: table_unreg
    generic map (IN_WIDTH   => 8,
                 OUT_WIDTH  => 12,
                 TABLE_FILE => "p3m8_12.mif"
    )
    port map(indata  => entry4(8 downto 1),
             outdata => ginv5,
             clk     => clk
    );

  -- index 0xf3 (1111 0011) indicate r/eta
  -- at which 12-bit 1/g table for P3M overflows.
  gnz4 <= '1' when (entry4(8 downto 1) < "11110011")
         else '0';

  process(clk) begin
    if(clk'event and clk='1') then
      gnz5 <= gnz4;
    end if;
  end process;

  ginv <= gnz5 & "000" & ginv5;

end rtl;

--
-- template: conv_log_to_float_17_8_17_9
--
-- convert a 17-bit floating-point to a 17-bit log
-- using a table "l8tf8.mif".
--
-- log:
-- 1     1         7            8
-- sign  non-zero  exponent     mantissa
--                 (no offset)
--
-- float:
-- 1     7            9
-- sign  exponent     mantissa
--       (with offset)
--
-- pipeline delay: 2
--
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity conv_log_to_float_17_8_17_9 is
  port (src    : in  std_logic_vector(16 downto 0);
        dst    : out std_logic_vector(16 downto 0);
        clk    : in  std_logic);
end conv_log_to_float_17_8_17_9;

architecture rtl of conv_log_to_float_17_8_17_9 is

  component table_reg
    generic (IN_WIDTH: integer ;
             OUT_WIDTH: integer ;
             TABLE_FILE: string);
    port (indata : in std_logic_vector(IN_WIDTH-1 downto 0);
         outdata : out std_logic_vector(OUT_WIDTH-1 downto 0);
         clk : in std_logic);
  end component;

  signal sign0, sign1, sign2: std_logic;
  signal nz0, nz1, nz2: std_logic;
  signal e0, e1, e2: std_logic_vector(6 downto 0);
  signal m0, m2: std_logic_vector(7 downto 0);
  signal fpdata2: std_logic_vector(16 downto 0);

begin
  -- stage 0
  sign0 <= src(16);                 -- sign
  nz0   <= src(15);                 -- non zero
  e0    <= src(14 downto 8);        -- exponent
  m0    <= src( 7 downto 0);        -- mantissa

  -- stage 1
  process(clk)
  begin
    if(clk'event and clk='1') then
      sign1 <= sign0;
      sign2 <= sign1;
      nz1 <= nz0;
      nz2 <= nz1;
      e1 <= e0;
      e2 <= e1;
    end if;
  end process;

  u1: table_reg generic map(IN_WIDTH=>8,OUT_WIDTH=>8,TABLE_FILE=>"l8tf8.mif")
    port map (indata=>m0,outdata=>m2,clk=>clk);

  fpdata2(16) <= sign2;
  fpdata2(15 downto 9) <= e2;
  fpdata2(8 downto 0) <= m2 & "0";

  with nz2 select
    dst <=
      fpdata2                      when '1',
      conv_std_logic_vector(0, 17) when others;

end rtl;
