-------------------------------------------------------------------------------
--
--
-- primitive components used by pg-modules.
--
--
-------------------------------------------------------------------------------

--
-- table_unreg: rom-table with unregistered address input
--
-- pipeline delay: 1
--
library ieee;
use ieee.std_logic_1164.all;

entity table_unreg is
  generic (IN_WIDTH   : integer;
           OUT_WIDTH  : integer;
           TABLE_FILE : string
  );
  port (indata  : in  std_logic_vector(IN_WIDTH-1 downto 0);
        outdata : out std_logic_vector(OUT_WIDTH-1 downto 0);
        clk     : in  std_logic
  );
end table_unreg;

architecture rtl of table_unreg is

  component lpm_rom
    generic (LPM_WIDTH           : POSITIVE;
             LPM_WIDTHAD         : POSITIVE;
             LPM_ADDRESS_CONTROL : STRING;
             LPM_FILE            : STRING
    );
    port (address  : in  std_logic_vector(LPM_WIDTHAD-1 downto 0);
          outclock : in  std_logic;
          q        : out std_logic_vector(LPM_WIDTH-1 downto 0)
    );
  end component;

begin

  u1: lpm_rom
    generic map (LPM_WIDTH           => OUT_WIDTH,
                 LPM_WIDTHAD         => IN_WIDTH,
                 LPM_ADDRESS_CONTROL => "UNREGISTERED",
                 LPM_FILE            => TABLE_FILE
    )
    port map (address  => indata,
              outclock => clk,
              q        => outdata
    );

end rtl;

--
-- table_reg: rom-table with registered address input
--
-- pipeline delay: 2
--
library ieee;
use ieee.std_logic_1164.all;

entity table_reg is
  generic (IN_WIDTH   : integer;
           OUT_WIDTH  : integer;
           TABLE_FILE : string
  );
  port (indata  : in  std_logic_vector(IN_WIDTH-1 downto 0);
        outdata : out std_logic_vector(OUT_WIDTH-1 downto 0);
        clk     : in  std_logic
  );
end table_reg;

architecture rtl of table_reg is

  component lpm_rom
    generic (LPM_WIDTH           : POSITIVE;
             LPM_WIDTHAD         : POSITIVE;
             LPM_ADDRESS_CONTROL : STRING;
             LPM_FILE            : STRING
    );
    port (address  : in  std_logic_vector(LPM_WIDTHAD-1 downto 0);
          inclock  : in  std_logic;
          outclock : in  std_logic;
          q        : out std_logic_vector(LPM_WIDTH-1 downto 0)
    );
  end component;

begin

  u1: lpm_rom generic map (LPM_WIDTH           => OUT_WIDTH,
                           LPM_WIDTHAD         => IN_WIDTH,
                           LPM_ADDRESS_CONTROL => "REGISTERED",
                           LPM_FILE            =>TABLE_FILE
               )
    port map (address  => indata,
              inclock  => clk,
              outclock => clk,
              q        => outdata
    );

end rtl;

--
-- lcell_lut: lcell primitive in normal mode.
--
-- MASK:
-- x3, x2, x1, x0  z
--  0,  0,  0,  0  MASK( 0)
--  0,  0,  0,  1  MASK( 1)
--  0,  0,  1,  0  MASK( 2)
--  0,  0,  1,  1  MASK( 3)
--  0,  1,  0,  0  MASK( 4)
--  0,  1,  0,  1  MASK( 5)
--  .....................
--  1,  1,  1,  1  MASK(16)
--
-- FF:
--   0: output 'z' is not registered.
--   1: output 'z' is registered.
--
library ieee;
use ieee.std_logic_1164.all;
library wysiwyg;
use wysiwyg.stratix_components.all;

entity lcell_lut is
  generic(MASK: string  := "ffff";
            FF: integer := 0);
    port(x : in std_logic_vector(3 downto 0);
         z : out std_logic;
         clk : in std_logic);
end lcell_lut;

architecture rtl of lcell_lut is

component stratix_lcell
  generic
  (
    operation_mode : string := "normal";
    synch_mode     : string := "off";
    register_cascade_mode : string := "off";
    sum_lutc_input : string := "datac";
    lut_mask       : string := "0000";
    power_up       : string := "low";
    cin0_used      : string := "false";
    cin1_used      : string := "false";
    cin_used       : string := "false";
    output_mode    : string := "comb_and_reg"
  );
  port (clk: in std_logic := '0';
    dataa  : in std_logic := '1';
    datab  : in std_logic := '1';
    datac  : in std_logic := '1';
    datad  : in std_logic := '1';
--  aclr   : in std_logic := '0';
--  aload  : in std_logic := '0';
--  sclr   : in std_logic := '0';
--  sload  : in std_logic := '0';
    ena    : in std_logic := '1';
--  cin    : in std_logic := '0';
--  cin0   : in std_logic := '0';
--  cin1   : in std_logic := '1';
--  inverta   : in std_logic := '0';
--  regcascin : in std_logic := '0';
    combout   : out std_logic;
    regout    : out std_logic);
--  cout   : out std_logic;
--  cout0  : out std_logic;
--  cout1  : out std_logic);
end component;

begin
ifgen0: if (FF=0) generate
  u0: stratix_lcell generic map(lut_mask=>MASK)
      port map(dataa=>x(0),datab=>x(1),datac=>x(2),datad=>x(3),
               clk=>clk,combout=>z);
end generate;
ifgen1: if (FF>0) generate
  u0: stratix_lcell generic map(lut_mask=>MASK)
      port map(dataa=>x(0),datab=>x(1),datac=>x(2),datad=>x(3),
               clk=>clk,regout=>z);
end generate;
end rtl;

--
-- lcell_lut: lcell primitive in arithmetic mode.
--
-- MASK:
-- ci, y, x  z       , co
--  0, 0, 0  MASK(8) , MASK(0)
--  0, 0, 1  MASK(9) , MASK(1)
--  0, 1, 0  MASK(10), MASK(2)
--  0, 1, 1  MASK(11), MASK(3)
--  1, 0, 0  MASK(12), MASK(4)
--  1, 0, 1  MASK(13), MASK(5)
--  1, 1, 0  MASK(14), MASK(6)
--  1, 1, 1  MASK(15), MASK(7)
--
-- FF:
--   0: output 'z' is not registered.
--   1: output 'z' is registered.
--
library ieee;
use ieee.std_logic_1164.all;
library wysiwyg;
use wysiwyg.stratix_components.all;

entity lcell_arithmetic is
  generic (MASK: string;
            FF: integer);
    port (x,y,ci : in std_logic;
         z,co : out std_logic;
         clk : in std_logic);
end lcell_arithmetic;

architecture rtl of lcell_arithmetic is

component stratix_lcell
   generic
  (
   operation_mode    : string := "arithmetic";
   synch_mode : string := "off";
   register_cascade_mode   : string := "off";
-- sum_lutc_input : string := "datac";
   lut_mask       : string := "96E8";
   power_up       : string := "low";
   cin0_used      : string := "false";
   cin1_used      : string := "false";
   cin_used       : string := "true";
   output_mode    : string := "comb_and_reg"
  );
 port (clk : in std_logic := '0';
   dataa   : in std_logic := '1';
   datab   : in std_logic := '1';
-- datac   : in std_logic := '1';
-- datad   : in std_logic := '1';
-- aclr    : in std_logic := '0';
-- aload   : in std_logic := '0';
-- sclr    : in std_logic := '0';
-- sload   : in std_logic := '0';
   ena     : in std_logic := '1';
   cin     : in std_logic := '0';
-- cin0    : in std_logic := '0';
-- cin1    : in std_logic := '1';
-- inverta : in std_logic := '0';
-- regcascin : in std_logic := '0';
   combout   : out std_logic;
   regout    : out std_logic;
   cout   : out std_logic);
-- cout0  : out std_logic;
-- cout1  : out std_logic);
end component;
begin

ifgen0: if (FF=0) generate
  u0: stratix_lcell generic map(lut_mask=>MASK)
      port map(dataa=>x,datab=>y,cin=>ci,combout=>z,cout=>co,clk=>clk);
end generate;
ifgen1: if (FF>0) generate
  u0: stratix_lcell generic map(lut_mask=>MASK)
      port map(dataa=>x,datab=>y,cin=>ci,regout=>z,cout=>co,clk=>clk);
end generate;

end rtl;

--
-- round_ubf1: perform unbiased force-1 rounding
--
-- pipeline delay: 0
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity round_ubf1 is
  generic (IN_WIDTH   : integer;
           OUT_WIDTH  : integer
  );
  port (indata  : in  std_logic_vector(IN_WIDTH-1 downto 0);
        outdata : out std_logic_vector(OUT_WIDTH-1 downto 0)
  );
end round_ubf1;

architecture rtl of round_ubf1 is

begin

-- 
-- an example for iw=20, ow=10
-- 
-- od(9 downto 1) <= id(19 downto 11)
-- with id(10 downto 0) select
-- od(0) <= '0' when conv_std_logic_vector(0, 11)
--       <= '1' when others;
--
  
  outdata(OUT_WIDTH-1 downto 1) <= indata(IN_WIDTH-1 downto IN_WIDTH-OUT_WIDTH+1);
  with indata(IN_WIDTH-OUT_WIDTH downto 0) select
    outdata(0) <=
    '0' when conv_std_logic_vector(0, IN_WIDTH-OUT_WIDTH+1),
    '1' when others;

end rtl;

--
-- mul_int: int multiplier
-- using dedicated DSP blocks of devices such as CycloneII and ArriaGX.
-- width of multiplier and multiplicant must be the same.
--
-- based on lpm wrapper generated by QuartusII MegaWizard.
--
library ieee;
use ieee.std_logic_1164.all;

library lpm;
use lpm.lpm_components.all;

entity mul_int is
  generic (
    WIDTH : integer;
    DELAY : integer := 2
  );
  port (
    clock        : in  std_logic;
    dataa        : in  std_logic_vector(WIDTH-1 downto 0);
    datab        : in  std_logic_vector(WIDTH-1 downto 0);
    result       : out std_logic_vector(WIDTH*2-1 downto 0)
  );
end mul_int;

architecture syn of mul_int is

  signal sub_wire0    : std_logic_vector(WIDTH*2-1 downto 0);

  component lpm_mult
    generic (
      lpm_hint           : string;
      lpm_pipeline       : natural;
      lpm_representation : string;
      lpm_type           : string;
      lpm_widtha         : natural;
      lpm_widthb         : natural;
      lpm_widthp         : natural;
      lpm_widths         : natural
    );
    port (
      dataa    : in std_logic_vector(WIDTH-1 downto 0);
      datab    : in std_logic_vector(WIDTH-1 downto 0);
      clock    : in std_logic ;
      result   : out std_logic_vector(WIDTH*2-1 downto 0)
    );
  end component;

begin
    result <= sub_wire0(WIDTH*2-1 downto 0);

    lpm_mult_component : lpm_mult
    generic map (
        lpm_hint           => "DEDICATED_MULTIPLIER_CIRCUITRY=YES,MAXIMIZE_SPEED=5",
        lpm_pipeline       => DELAY,
        lpm_representation => "UNSIGNED",
        lpm_type           => "LPM_MULT",
        lpm_widtha         => WIDTH,
        lpm_widthb         => WIDTH,
        lpm_widthp         => WIDTH*2,
        lpm_widths         => 1
    )
    port map (
        dataa  => dataa,
        datab  => datab,
        clock  => clock,
        result => sub_wire0
    );

end syn;

--
-- mul_int2: int multiplier
-- using dedicated DSP blocks of devices such as CycloneII and ArriaGX.
-- multiplier and multiplicant may have different width.
--
-- based on lpm wrapper generated by QuartusII MegaWizard.
--
library ieee;
use ieee.std_logic_1164.all;

library lpm;
use lpm.lpm_components.all;

entity mul_int2 is
  generic (
    WIDTHA : integer;
    WIDTHB : integer;
    DELAY  : integer := 2
  );
  port (
    clock        : in  std_logic;
    dataa        : in  std_logic_vector(WIDTHA-1 downto 0);
    datab        : in  std_logic_vector(WIDTHB-1 downto 0);
    result       : out std_logic_vector(WIDTHA+WIDTHB-1 downto 0)
  );
end mul_int2;

architecture syn of mul_int2 is

  signal sub_wire0    : std_logic_vector(WIDTHA+WIDTHB-1 downto 0);

  component lpm_mult
    generic (
      lpm_hint           : string;
      lpm_pipeline       : natural;
      lpm_representation : string;
      lpm_type           : string;
      lpm_widtha         : natural;
      lpm_widthb         : natural;
      lpm_widthp         : natural;
      lpm_widths         : natural
    );
    port (
      dataa    : in std_logic_vector(WIDTHA-1 downto 0);
      datab    : in std_logic_vector(WIDTHB-1 downto 0);
      clock    : in std_logic ;
      result   : out std_logic_vector(WIDTHA+WIDTHB-1 downto 0)
    );
  end component;

begin
    result <= sub_wire0(WIDTHA+WIDTHB-1 downto 0);

    lpm_mult_component : lpm_mult
    generic map (
        lpm_hint           => "DEDICATED_MULTIPLIER_CIRCUITRY=YES,MAXIMIZE_SPEED=5",
        lpm_pipeline       => DELAY,
        lpm_representation => "UNSIGNED",
        lpm_type           => "LPM_MULT",
        lpm_widtha         => WIDTHA,
        lpm_widthb         => WIDTHB,
        lpm_widthp         => WIDTHA+WIDTHB,
        lpm_widths         => 1
    )
    port map (
        dataa  => dataa,
        datab  => datab,
        clock  => clock,
        result => sub_wire0
    );

end syn;

-------------------------------------------------------------------------------
--
--
-- pg-modules generated by pg2vhdl based on .pgdl description
-- and .template.vhd templates.
--
--
-------------------------------------------------------------------------------

