library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;

-- KFCR board_info register initial value:
--
-- (31:28) product ID      0x1:GRAPE-7 0x2:GRAPE-7E 0x3:GRAPE-DR
-- (27:24) model No        0x1:model100 0x2:model300D 0x3:model300 0x6:model600
-- (23:20) backend design  0x0:empty 0x1:G5X 0x2:G6X
-- (19:18  not used
-- (17)    grav potential  0x0:not available 0x1:available
-- (16)    neighbor list   0x0:not available 0x1:available
-- (15)    P3M cutoff      0x0:not available 0x1:available
-- (14)    eps2 format     0x0:logarithmic 0x1:floating point
-- (13:10) reserved for use in hib entity
-- (9:8)   j-mem size      0x0:2k 0x1:4k 0x2:8k 0x3:16k in particles
-- (7:0)   # of pipes      min:0x00 max:0xff
--
entity backend is
  generic(
    -- for backend entity internal use
    JDATA_WIDTH        : integer := <JDATA_WIDTH>;
    JDATA_DEPTH        : integer := <JDATA_DEPTH>;
    JMEM_BLOCKS        : integer := <JMEM_BLOCKS>;

    -- board info register initial value
    KFCR_PRODUCT_ID    : integer := <PRODUCTID>;
    G7_MODEL_NO        : integer := <MODELID>;
    G7_BACKEND         : integer := <BACKENDID>;
    POTENTIAL          : integer := 0;
    NBLIST             : integer := 1;
    P3MCUTOFF          : integer := 1;
    EPS2_IN            : integer := 1;
    J_MEM_SIZE         : integer := <JDATA_DEPTH> - 11;
    NPIPES             : integer := <NPIPES>
  );
  port (
    -- DMA signals
    hib_we             : in std_logic;
    hib_data           : in std_logic_vector(63 downto 0);
    backend_we         : out std_logic;
    backend_data       : out std_logic_vector(63 downto 0);
    backend_run        : out std_logic;

    board_info         : out std_logic_vector(31 downto 0); -- board info register initial value
    o_cid              : in std_logic_vector(2 downto 0);   -- chip id

    -- global signals
    hib_clk: in std_logic;
    backend_clk0: in std_logic;
    rst: in std_logic                   -- negative logic
    );
end backend;

architecture hierarchy of backend is

  component pg_ctl
    generic (
      JDATA_WIDTH : integer;
      JDATA_DEPTH : integer;
      JMEM_BLOCKS : integer
    );
    port(
      hib_we       : in std_logic;
      hib_data     : in std_logic_vector(63 downto 0);
      backend_we   : out std_logic;
      backend_data : out std_logic_vector(63 downto 0);
      rst          : in std_logic;
      clk          : in std_logic;
      p_jdata      : out std_logic_vector(JDATA_WIDTH-1 downto 0);
      p_run        : out std_logic;
      p_runret     : in std_logic;
      p_bcast      : out std_logic;
      p_we         : out std_logic;
      p_adri       : out std_logic_vector(11+4 downto 0);
      p_datai      : out std_logic_vector(63 downto 0);
      p_adro       : out std_logic_vector(11+4 downto 0);
      p_datao      : in std_logic_vector(63 downto 0)
      );
  end component;

  component pg_pipe
    generic(JDATA_WIDTH : integer;
            NPIPES      : integer);
    port(
      p_jdata  : in std_logic_vector(JDATA_WIDTH-1 downto 0);
      p_run    : in std_logic;
      p_we     : in std_logic;
      p_adri   : in std_logic_vector(11+4 downto 0);
      p_datai  : in std_logic_vector(63 downto 0);
      p_adro   : in std_logic_vector(11+4 downto 0);
      p_datao  : out std_logic_vector(63 downto 0);
      p_runret : out std_logic;
      p_bcast  : in std_logic;
      p_cid    : in  std_logic_vector(2 downto 0);
      rst      : in std_logic;
      clk      : in std_logic );
  end component;

  component pg_ifbuf
    port(
      hib_we: in std_logic;
      hib_data: in std_logic_vector(63 downto 0);
      backend_we: out std_logic;
      backend_data: out std_logic_vector(63 downto 0);
      hib_we_buf: out std_logic;
      hib_data_buf: out std_logic_vector(63 downto 0);
      backend_we_buf: in std_logic;
      backend_data_buf: in std_logic_vector(63 downto 0);
      clka: in std_logic;
      clkb: in std_logic;
      rst: in std_logic);
  end component;

  signal p_jdata: std_logic_vector(JDATA_WIDTH-1 downto 0);
  signal p_run : std_logic;
  signal p_we : std_logic;
  signal p_adri : std_logic_vector(11+4 downto 0);
  signal p_datai : std_logic_vector(63 downto 0);
  signal p_adro : std_logic_vector(11+4 downto 0);
  signal p_datao : std_logic_vector(63 downto 0);
  signal p_runret : std_logic;
  signal p_bcast : std_logic;

  signal hib_we_buf: std_logic;
  signal hib_data_buf: std_logic_vector(63 downto 0);
  signal backend_we_buf: std_logic;
  signal backend_data_buf: std_logic_vector(63 downto 0);

begin

  uifbuf: pg_ifbuf
    port map(hib_we           => hib_we,
             hib_data         => hib_data,
             backend_we       => backend_we,
             backend_data     => backend_data,
             hib_we_buf       => hib_we_buf,
             hib_data_buf     => hib_data_buf,
             backend_we_buf   => backend_we_buf,
             backend_data_buf => backend_data_buf,
             clka             => hib_clk,
             clkb             => backend_clk0,
             rst              => rst);

  uctl: pg_ctl
    generic map(JDATA_WIDTH => JDATA_WIDTH,
                JDATA_DEPTH => JDATA_DEPTH,
                JMEM_BLOCKS => JMEM_BLOCKS)
    port map(hib_we       => hib_we_buf,
             hib_data     => hib_data_buf,
             backend_we   => backend_we_buf,
             backend_data => backend_data_buf,
             clk          => backend_clk0,
             rst          => rst,
             p_jdata      => p_jdata,
             p_run        => p_run,
             p_runret     => p_runret,
             p_we         => p_we,
             p_adri       => p_adri,
             p_datai      => p_datai,
             p_adro       => p_adro,
             p_datao      => p_datao,
             p_bcast      => p_bcast);

  upipe : pg_pipe
    generic map (JDATA_WIDTH => JDATA_WIDTH,
                 NPIPES      => NPIPES)
    port map(p_jdata  => p_jdata,
             p_run    => p_run,
             p_we     => p_we,
             p_adri   => p_adri,
             p_datai  => p_datai,
             p_adro   => p_adro,
             p_datao  => p_datao,
             p_runret => p_runret,
             p_bcast  => p_bcast,
             p_cid    => o_cid,
             rst      => rst,
             clk      => backend_clk0);

  backend_run <= p_run;

  board_info <=
      conv_std_logic_vector(KFCR_PRODUCT_ID, 4) &
      conv_std_logic_vector(G7_MODEL_NO, 4) &
      conv_std_logic_vector(G7_BACKEND, 4) &
      conv_std_logic_vector(0, 2) &
      conv_std_logic_vector(POTENTIAL, 1) &
      conv_std_logic_vector(NBLIST, 1) &
      conv_std_logic_vector(P3MCUTOFF, 1) &
      conv_std_logic_vector(EPS2_IN, 1) &
      conv_std_logic_vector(0, 2) &
      conv_std_logic_vector(0, 2) &
      conv_std_logic_vector(J_MEM_SIZE, 2) &
      conv_std_logic_vector(NPIPES, 8);

end hierarchy;
